/**
 * Cast Conductor Proprietary License v5
 * SPDX-License-Identifier: LicenseRef-CastConductor-Proprietary-v5
 * 
 * Copyright (c) 2025 CastConductor.com. All Rights Reserved.
 * 
 * This file is part of Cast Conductor ("Software"). The Software and its source
 * code constitute proprietary, confidential, and trade secret information of
 * CastConductor.com ("Company"). Any access or use is governed strictly by the
 * Cast Conductor Proprietary License v5 ("License"). By installing, copying,
 * accessing, compiling, or otherwise using the Software you agree to be bound by
 * all terms of the License. If you do not agree, you must cease use immediately.
 * 
 * Key Terms (Summary – see full License for binding terms):
 *  1. No Redistribution: You may not publish, distribute, sublicense, rent,
 *     lease, transfer, sell, or otherwise make the Software (or any derivative)
 *     available to any third party without prior written consent of Company.
 *  2. No Modification: Modification, reverse engineering, decompilation, or
 *     disassembly is prohibited except to the limited extent expressly permitted
 *     by applicable law that cannot be contractually waived.
 *  3. Confidentiality: Treat all source code and related artifacts as Company
 *     Confidential Information. Maintain at least the same degree of care as for
 *     your own confidential materials, and not less than reasonable care.
 *  4. No Patent License: No express or implied patent rights are granted. Future
 *     patents (if any) are fully reserved.
 *  5. No Trademark License: Company names, marks, and logos may not be used
 *     without prior written permission.
 *  6. Limited Internal Use: Use is limited solely to internal evaluation and
 *     operation of licensed Cast Conductor deployments. Commercial hosting or
 *     resale as a service requires a separate written agreement.
 *  7. Telemetry & License Validation: The Software may periodically transmit a
 *     hashed installation identifier, domain (or site ID), plugin/app version,
 *     and a truncated (non-reversible) fragment of the license key solely to
 *     validate activation status and enforce licensing. This minimal "phone home"
 *     check contains no personal or content data. If optional telemetry is later
 *     introduced it will be limited to aggregate operational metrics (no PII),
 *     fully documented, and optionally disableable per published instructions.
 *  8. Third-Party Components: The Software may include open source components
 *     covered by their own licenses. See THIRD-PARTY-NOTICES.md. Those licenses
 *     govern their respective components; this License governs all remaining code.
 *  9. Export Compliance: You are responsible for compliance with all applicable
 *     export control and sanctions laws.
 * 10. Warranty Disclaimer: THE SOFTWARE IS PROVIDED "AS IS" WITHOUT WARRANTY OF
 *     ANY KIND, EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO MERCHANTABILITY,
 *     FITNESS FOR A PARTICULAR PURPOSE, TITLE, AND NON-INFRINGEMENT.
 * 11. Limitation of Liability: IN NO EVENT WILL COMPANY OR AUTHORS BE LIABLE FOR
 *     ANY INDIRECT, SPECIAL, INCIDENTAL, CONSEQUENTIAL, EXEMPLARY, OR PUNITIVE
 *     DAMAGES, OR LOST PROFITS, EVEN IF ADVISED OF THE POSSIBILITY.
 * 12. Acceptance: Use of the Software constitutes acceptance of the License.
 * 13. Enforcement: Unauthorized reproduction or distribution may result in civil
 *     and criminal penalties and will be prosecuted to the maximum extent allowed
 *     by law.
 * 
 * Authoritative EULA: EULA-v5.1.md (repository root – private) and https://castconductor.com/eula
 * Precedence: If this summary conflicts with the EULA, the EULA governs.
 * Revision: Current EULA revision v5.1 (subject to update; check EULA for current enterprise thresholds).
 * 
 * Full License text available from: licensing@castconductor.com
 * Security reports: security@castconductor.com
 * Commercial inquiries: licensing@castconductor.com
 * 
 * END OF HEADER
 */
class CastConductorShoutoutModeration {
    constructor() {
        this.currentPage = 1;
        this.perPage = 20;
        this.selectedShoutouts = new Set();
        this.settings = {};
        this.refreshInterval = null;
        
        this.init();
    }

    /**
     * Initialize the moderation interface
     */
    init() {
        this.loadModerationSettings();
        this.setupEventListeners();
        this.setupTabNavigation();
        this.loadPendingShoutouts();
        this.loadModerationStats();
        this.setupAutoRefresh();
    }

    /**
     * Setup tab navigation
     */
    setupTabNavigation() {
        const navTabs = document.querySelectorAll('.moderation-nav-tab');
        const tabPanes = document.querySelectorAll('.moderation-tab-pane');
        
        navTabs.forEach(tab => {
            tab.addEventListener('click', () => {
                const targetTab = tab.dataset.tab;
                
                // Remove active class from all tabs and panes
                navTabs.forEach(t => t.classList.remove('active'));
                tabPanes.forEach(p => p.classList.remove('active'));
                
                // Add active class to clicked tab and corresponding pane
                tab.classList.add('active');
                const targetPane = document.getElementById(`${targetTab}-tab`);
                if (targetPane) {
                    targetPane.classList.add('active');
                }
            });
        });
    }

    /**
     * Setup event listeners
     */
    setupEventListeners() {
        // Pagination controls
        document.addEventListener('click', (e) => {
            if (e.target.classList.contains('moderation-page-btn')) {
                this.currentPage = parseInt(e.target.dataset.page);
                this.loadPendingShoutouts();
            }
        });

        // Individual shoutout actions
        document.addEventListener('click', (e) => {
            if (e.target.classList.contains('approve-shoutout')) {
                const id = e.target.dataset.id;
                this.approveShoutout(id);
            } else if (e.target.classList.contains('reject-shoutout')) {
                const id = e.target.dataset.id;
                this.showRejectModal(id);
            }
        });

        // Bulk actions
        const bulkActionSelect = document.getElementById('bulk-action-select');
        const bulkActionBtn = document.getElementById('bulk-action-btn');
        
        if (bulkActionBtn) {
            bulkActionBtn.addEventListener('click', () => {
                const action = bulkActionSelect?.value;
                if (action && this.selectedShoutouts.size > 0) {
                    this.showBulkActionModal(action);
                }
            });
        }

        // Select all checkbox
        const selectAllCheckbox = document.getElementById('select-all-shoutouts');
        if (selectAllCheckbox) {
            selectAllCheckbox.addEventListener('change', (e) => {
                this.toggleSelectAll(e.target.checked);
            });
        }

        // Individual checkboxes
        document.addEventListener('change', (e) => {
            if (e.target.classList.contains('shoutout-checkbox')) {
                const id = e.target.dataset.id;
                if (e.target.checked) {
                    this.selectedShoutouts.add(id);
                } else {
                    this.selectedShoutouts.delete(id);
                }
                this.updateBulkActionsState();
            }
        });

        // Settings form
        const settingsForm = document.getElementById('moderation-settings-form');
        if (settingsForm) {
            settingsForm.addEventListener('submit', (e) => {
                e.preventDefault();
                this.saveModerationSettings();
            });
        }

        // Test reCAPTCHA button
        const testRecaptchaBtn = document.getElementById('test-recaptcha-btn');
        if (testRecaptchaBtn) {
            testRecaptchaBtn.addEventListener('click', () => {
                this.testRecaptchaConfiguration();
            });
        }

        // Refresh button
        const refreshBtn = document.getElementById('refresh-moderation-btn');
        if (refreshBtn) {
            refreshBtn.addEventListener('click', () => {
                this.refreshAll();
            });
        }

        // Auto-refresh toggle
        const autoRefreshToggle = document.getElementById('auto-refresh-toggle');
        if (autoRefreshToggle) {
            autoRefreshToggle.addEventListener('change', (e) => {
                if (e.target.checked) {
                    this.setupAutoRefresh();
                } else {
                    this.clearAutoRefresh();
                }
            });
        }
    }

    /**
     * Load pending shoutouts for moderation
     */
    async loadPendingShoutouts() {
        try {
            this.showLoader('Loading pending shoutouts...');

            const response = await fetch(
                `${castconductorModerationAjax.rest_url}castconductor/v5/shoutout-moderation/pending?page=${this.currentPage}&per_page=${this.perPage}`,
                {
                    method: 'GET',
                    headers: {
                        'X-WP-Nonce': castconductorModerationAjax.nonce
                    }
                }
            );

            const data = await response.json();
            if (data.success) {
                this.renderPendingShoutouts(data.shoutouts);
                this.renderPagination(data.pagination);
                this.selectedShoutouts.clear();
                this.updateBulkActionsState();
            } else {
                this.showNotification('Failed to load pending shoutouts', 'error');
            }
        } catch (error) {
            console.error('Failed to load pending shoutouts:', error);
            this.showNotification('Failed to load pending shoutouts', 'error');
        } finally {
            this.hideLoader();
        }
    }

    /**
     * Render pending shoutouts list
     */
    renderPendingShoutouts(shoutouts) {
        const container = document.getElementById('pending-shoutouts-list');
        if (!container) return;

        if (shoutouts.length === 0) {
            container.innerHTML = `
                <div class="no-shoutouts">
                    <div class="dashicons dashicons-yes-alt"></div>
                    <h3>No pending shoutouts</h3>
                    <p>All shoutouts have been reviewed. Great job!</p>
                </div>
            `;
            return;
        }

        container.innerHTML = shoutouts.map(shoutout => `
            <div class="shoutout-item ${this.getShoutoutRiskClass(shoutout.moderation_context)}" data-id="${shoutout.id}">
                <div class="shoutout-header">
                    <label class="shoutout-checkbox-label">
                        <input type="checkbox" class="shoutout-checkbox" data-id="${shoutout.id}">
                        <span class="checkmark"></span>
                    </label>
                    <div class="shoutout-meta">
                        <span class="shoutout-id">#${shoutout.id}</span>
                        <span class="shoutout-date">${this.formatDate(shoutout.created_at)}</span>
                        ${this.renderRiskBadge(shoutout.moderation_context)}
                    </div>
                    <div class="shoutout-actions">
                        <button type="button" class="button button-primary approve-shoutout" data-id="${shoutout.id}">
                            <span class="dashicons dashicons-yes"></span> Approve
                        </button>
                        <button type="button" class="button button-secondary reject-shoutout" data-id="${shoutout.id}">
                            <span class="dashicons dashicons-no"></span> Reject
                        </button>
                    </div>
                </div>
                
                <div class="shoutout-content">
                    <div class="shoutout-details">
                        <div class="detail-group">
                            <label>Name:</label>
                            <span class="shoutout-name">${this.escapeHtml(shoutout.content.name || 'Anonymous')}</span>
                        </div>
                        <div class="detail-group">
                            <label>Location:</label>
                            <span class="shoutout-location">${this.escapeHtml(shoutout.content.location || 'Not provided')}</span>
                        </div>
                    </div>
                    
                    <div class="shoutout-message">
                        <label>Message:</label>
                        <div class="message-text">${this.escapeHtml(shoutout.content.message || '')}</div>
                    </div>
                    
                    ${this.renderModerationFlags(shoutout.moderation_context)}
                </div>
            </div>
        `).join('');
    }

    /**
     * Render pagination controls
     */
    renderPagination(pagination) {
        const container = document.getElementById('moderation-pagination');
        if (!container) return;

        if (pagination.total_pages <= 1) {
            container.innerHTML = '';
            return;
        }

        let paginationHTML = '<div class="tablenav-pages">';
        paginationHTML += `<span class="displaying-num">${pagination.total} items</span>`;
        
        if (pagination.page > 1) {
            paginationHTML += `<button class="button moderation-page-btn" data-page="${pagination.page - 1}">‹ Previous</button>`;
        }
        
        // Page numbers
        const startPage = Math.max(1, pagination.page - 2);
        const endPage = Math.min(pagination.total_pages, pagination.page + 2);
        
        for (let i = startPage; i <= endPage; i++) {
            const activeClass = i === pagination.page ? 'current' : '';
            paginationHTML += `<button class="button moderation-page-btn ${activeClass}" data-page="${i}">${i}</button>`;
        }
        
        if (pagination.page < pagination.total_pages) {
            paginationHTML += `<button class="button moderation-page-btn" data-page="${pagination.page + 1}">Next ›</button>`;
        }
        
        paginationHTML += '</div>';
        container.innerHTML = paginationHTML;
    }

    /**
     * Approve a shoutout
     */
    async approveShoutout(id) {
        try {
            const response = await fetch(
                `${castconductorModerationAjax.rest_url}castconductor/v5/shoutout-moderation/approve/${id}`,
                {
                    method: 'POST',
                    headers: {
                        'X-WP-Nonce': castconductorModerationAjax.nonce
                    }
                }
            );

            const data = await response.json();
            if (data.success) {
                this.showNotification('Shoutout approved successfully', 'success');
                this.removeShoutoutFromList(id);
                this.loadModerationStats();
            } else {
                this.showNotification('Failed to approve shoutout', 'error');
            }
        } catch (error) {
            console.error('Failed to approve shoutout:', error);
            this.showNotification('Failed to approve shoutout', 'error');
        }
    }

    /**
     * Show reject modal
     */
    showRejectModal(id) {
        const modal = document.getElementById('reject-modal');
        const reasonInput = document.getElementById('reject-reason');
        const confirmBtn = document.getElementById('confirm-reject');
        
        if (modal && reasonInput && confirmBtn) {
            reasonInput.value = '';
            modal.style.display = 'block';
            
            // Update confirm button handler
            confirmBtn.onclick = () => {
                const reason = reasonInput.value.trim() || 'Inappropriate content';
                this.rejectShoutout(id, reason);
                modal.style.display = 'none';
            };
        }
    }

    /**
     * Reject a shoutout
     */
    async rejectShoutout(id, reason) {
        try {
            const response = await fetch(
                `${castconductorModerationAjax.rest_url}castconductor/v5/shoutout-moderation/reject/${id}`,
                {
                    method: 'POST',
                    headers: {
                        'Content-Type': 'application/json',
                        'X-WP-Nonce': castconductorModerationAjax.nonce
                    },
                    body: JSON.stringify({ reason })
                }
            );

            const data = await response.json();
            if (data.success) {
                this.showNotification(`Shoutout rejected: ${reason}`, 'success');
                this.removeShoutoutFromList(id);
                this.loadModerationStats();
            } else {
                this.showNotification('Failed to reject shoutout', 'error');
            }
        } catch (error) {
            console.error('Failed to reject shoutout:', error);
            this.showNotification('Failed to reject shoutout', 'error');
        }
    }

    /**
     * Show bulk action modal
     */
    showBulkActionModal(action) {
        const modal = document.getElementById('bulk-action-modal');
        const actionText = document.getElementById('bulk-action-text');
        const reasonInput = document.getElementById('bulk-reason');
        const confirmBtn = document.getElementById('confirm-bulk-action');
        
        if (modal && actionText && confirmBtn) {
            const count = this.selectedShoutouts.size;
            actionText.textContent = `${action} ${count} shoutout${count > 1 ? 's' : ''}`;
            
            // Show/hide reason input
            if (reasonInput) {
                reasonInput.style.display = action === 'reject' ? 'block' : 'none';
                reasonInput.value = '';
            }
            
            modal.style.display = 'block';
            
            // Update confirm button handler
            confirmBtn.onclick = () => {
                const reason = reasonInput?.value.trim() || `Bulk ${action}`;
                this.performBulkAction(action, Array.from(this.selectedShoutouts), reason);
                modal.style.display = 'none';
            };
        }
    }

    /**
     * Perform bulk action
     */
    async performBulkAction(action, ids, reason) {
        try {
            this.showLoader(`Processing ${action} for ${ids.length} shoutouts...`);

            const response = await fetch(
                `${castconductorModerationAjax.rest_url}castconductor/v5/shoutout-moderation/bulk`,
                {
                    method: 'POST',
                    headers: {
                        'Content-Type': 'application/json',
                        'X-WP-Nonce': castconductorModerationAjax.nonce
                    },
                    body: JSON.stringify({ action, ids, reason })
                }
            );

            const data = await response.json();
            if (data.success) {
                this.showNotification(`Bulk ${action} completed: ${data.processed}/${data.total} processed`, 'success');
                this.loadPendingShoutouts();
                this.loadModerationStats();
            } else {
                this.showNotification(`Bulk ${action} failed`, 'error');
            }
        } catch (error) {
            console.error('Failed to perform bulk action:', error);
            this.showNotification(`Bulk ${action} failed`, 'error');
        } finally {
            this.hideLoader();
        }
    }

    /**
     * Load moderation statistics
     */
    async loadModerationStats() {
        try {
            const response = await fetch(
                `${castconductorModerationAjax.rest_url}castconductor/v5/shoutout-moderation/stats`,
                {
                    method: 'GET',
                    headers: {
                        'X-WP-Nonce': castconductorModerationAjax.nonce
                    }
                }
            );

            const data = await response.json();
            if (data.success) {
                this.renderModerationStats(data.stats);
            }
        } catch (error) {
            console.error('Failed to load moderation stats:', error);
        }
    }

    /**
     * Render moderation statistics
     */
    renderModerationStats(stats) {
        // All-time stats
        const allTimeContainer = document.getElementById('all-time-stats');
        if (allTimeContainer) {
            allTimeContainer.innerHTML = `
                <div class="stat-item pending">
                    <span class="stat-number">${stats.all_time.pending}</span>
                    <span class="stat-label">Pending</span>
                </div>
                <div class="stat-item approved">
                    <span class="stat-number">${stats.all_time.approved}</span>
                    <span class="stat-label">Approved</span>
                </div>
                <div class="stat-item rejected">
                    <span class="stat-number">${stats.all_time.rejected}</span>
                    <span class="stat-label">Rejected</span>
                </div>
                <div class="stat-item total">
                    <span class="stat-number">${stats.all_time.total}</span>
                    <span class="stat-label">Total</span>
                </div>
            `;
        }

        // Today's stats
        const todayContainer = document.getElementById('today-stats');
        if (todayContainer) {
            todayContainer.innerHTML = `
                <div class="stat-item pending">
                    <span class="stat-number">${stats.today.pending}</span>
                    <span class="stat-label">Pending Today</span>
                </div>
                <div class="stat-item approved">
                    <span class="stat-number">${stats.today.approved}</span>
                    <span class="stat-label">Approved Today</span>
                </div>
                <div class="stat-item rejected">
                    <span class="stat-number">${stats.today.rejected}</span>
                    <span class="stat-label">Rejected Today</span>
                </div>
            `;
        }
    }

    /**
     * Load moderation settings
     */
    async loadModerationSettings() {
        try {
            const response = await fetch(
                `${castconductorModerationAjax.rest_url}castconductor/v5/shoutout-moderation/settings`,
                {
                    method: 'GET',
                    headers: {
                        'X-WP-Nonce': castconductorModerationAjax.nonce
                    }
                }
            );

            const data = await response.json();
            if (data.success) {
                this.settings = data.settings;
                this.populateSettingsForm(data.settings);
            }
        } catch (error) {
            console.error('Failed to load moderation settings:', error);
        }
    }

    /**
     * Populate settings form
     */
    populateSettingsForm(settings) {
        const form = document.getElementById('moderation-settings-form');
        if (!form) return;

        // Set checkbox values
        const checkboxes = {
            'moderation-enabled': settings.moderation_enabled,
            'auto-approve': settings.auto_approve,
            'word-filter': settings.word_filter,
            'duplicate-check': settings.duplicate_check,
            'recaptcha-enabled': settings.recaptcha_enabled,
            'email-notifications': settings.email_notifications
        };

        Object.entries(checkboxes).forEach(([id, value]) => {
            const checkbox = document.getElementById(id);
            if (checkbox) {
                checkbox.checked = value;
            }
        });

        // Set text values
        const textFields = {
            'recaptcha-site-key': settings.recaptcha_site_key,
            'recaptcha-secret-key': settings.recaptcha_secret_key,
            'notification-email': settings.notification_email
        };

        Object.entries(textFields).forEach(([id, value]) => {
            const field = document.getElementById(id);
            if (field) {
                field.value = value || '';
            }
        });

        // Set threshold
        const thresholdField = document.getElementById('recaptcha-threshold');
        if (thresholdField) {
            thresholdField.value = settings.recaptcha_threshold || 0.5;
        }
    }

    /**
     * Save moderation settings
     */
    async saveModerationSettings() {
        try {
            const formData = new FormData(document.getElementById('moderation-settings-form'));
            const settings = {};

            // Convert form data to settings object
            settings.moderation_enabled = formData.get('moderation_enabled') === 'on';
            settings.auto_approve = formData.get('auto_approve') === 'on';
            settings.word_filter = formData.get('word_filter') === 'on';
            settings.duplicate_check = formData.get('duplicate_check') === 'on';
            settings.recaptcha_enabled = formData.get('recaptcha_enabled') === 'on';
            settings.email_notifications = formData.get('email_notifications') === 'on';
            settings.recaptcha_site_key = formData.get('recaptcha_site_key');
            settings.recaptcha_secret_key = formData.get('recaptcha_secret_key');
            settings.recaptcha_threshold = parseFloat(formData.get('recaptcha_threshold'));
            settings.notification_email = formData.get('notification_email');

            const response = await fetch(
                `${castconductorModerationAjax.rest_url}castconductor/v5/shoutout-moderation/settings`,
                {
                    method: 'POST',
                    headers: {
                        'Content-Type': 'application/json',
                        'X-WP-Nonce': castconductorModerationAjax.nonce
                    },
                    body: JSON.stringify(settings)
                }
            );

            const data = await response.json();
            if (data.success) {
                this.showNotification('Settings saved successfully', 'success');
                this.settings = settings;
            } else {
                this.showNotification('Failed to save settings', 'error');
            }
        } catch (error) {
            console.error('Failed to save settings:', error);
            this.showNotification('Failed to save settings', 'error');
        }
    }

    /**
     * Test reCAPTCHA configuration
     */
    async testRecaptchaConfiguration() {
        try {
            // This would normally use the actual reCAPTCHA widget
            // For testing purposes, we'll use a dummy token
            const testToken = 'test_token_' + Date.now();

            const response = await fetch(
                `${castconductorModerationAjax.rest_url}castconductor/v5/shoutout-moderation/test-recaptcha`,
                {
                    method: 'POST',
                    headers: {
                        'Content-Type': 'application/json',
                        'X-WP-Nonce': castconductorModerationAjax.nonce
                    },
                    body: JSON.stringify({ test_token: testToken })
                }
            );

            const data = await response.json();
            if (data.success) {
                const message = data.valid ? 
                    'reCAPTCHA configuration is working correctly' : 
                    'reCAPTCHA configuration has issues';
                this.showNotification(message, data.valid ? 'success' : 'warning');
            } else {
                this.showNotification('reCAPTCHA test failed', 'error');
            }
        } catch (error) {
            console.error('Failed to test reCAPTCHA:', error);
            this.showNotification('reCAPTCHA test failed', 'error');
        }
    }

    /**
     * Helper methods
     */
    
    getShoutoutRiskClass(context) {
        return `risk-${context.risk_level}`;
    }

    renderRiskBadge(context) {
        const riskColors = {
            low: 'green',
            medium: 'yellow',
            high: 'red'
        };
        const color = riskColors[context.risk_level] || 'gray';
        return `<span class="risk-badge risk-${context.risk_level}">${context.risk_level.toUpperCase()}</span>`;
    }

    renderModerationFlags(context) {
        if (!context.flags || context.flags.length === 0) {
            return '<div class="moderation-flags"><span class="flag-none">No issues detected</span></div>';
        }

        const flagsHTML = context.flags.map(flag => {
            const flagNames = {
                word_filter: 'Flagged Words',
                duplicate_content: 'Duplicate Content'
            };
            return `<span class="flag flag-${flag}">${flagNames[flag] || flag}</span>`;
        }).join('');

        return `<div class="moderation-flags">${flagsHTML}</div>`;
    }

    formatDate(dateString) {
        return new Date(dateString).toLocaleString();
    }

    escapeHtml(text) {
        const div = document.createElement('div');
        div.textContent = text;
        return div.innerHTML;
    }

    removeShoutoutFromList(id) {
        const item = document.querySelector(`.shoutout-item[data-id="${id}"]`);
        if (item) {
            item.remove();
        }
        this.selectedShoutouts.delete(id.toString());
        this.updateBulkActionsState();
    }

    toggleSelectAll(checked) {
        const checkboxes = document.querySelectorAll('.shoutout-checkbox');
        checkboxes.forEach(checkbox => {
            checkbox.checked = checked;
            const id = checkbox.dataset.id;
            if (checked) {
                this.selectedShoutouts.add(id);
            } else {
                this.selectedShoutouts.delete(id);
            }
        });
        this.updateBulkActionsState();
    }

    updateBulkActionsState() {
        const bulkActionBtn = document.getElementById('bulk-action-btn');
        const selectedCount = document.getElementById('selected-count');
        
        if (bulkActionBtn) {
            bulkActionBtn.disabled = this.selectedShoutouts.size === 0;
        }
        
        if (selectedCount) {
            selectedCount.textContent = this.selectedShoutouts.size;
        }
    }

    setupAutoRefresh() {
        this.clearAutoRefresh();
        this.refreshInterval = setInterval(() => {
            this.refreshAll();
        }, 30000); // Refresh every 30 seconds
    }

    clearAutoRefresh() {
        if (this.refreshInterval) {
            clearInterval(this.refreshInterval);
            this.refreshInterval = null;
        }
    }

    refreshAll() {
        this.loadPendingShoutouts();
        this.loadModerationStats();
    }

    showNotification(message, type = 'info') {
        // Create notification element
        const notification = document.createElement('div');
        notification.className = `moderation-notification moderation-notification-${type}`;
        notification.textContent = message;
        
        // Add to container
        const container = document.getElementById('moderation-notifications') || document.body;
        container.appendChild(notification);
        
        // Auto-remove after 5 seconds
        setTimeout(() => {
            if (notification.parentNode) {
                notification.parentNode.removeChild(notification);
            }
        }, 5000);
    }

    showLoader(message = 'Loading...') {
        const loader = document.getElementById('moderation-loader');
        if (loader) {
            loader.style.display = 'flex';
            const loaderText = loader.querySelector('.loader-text');
            if (loaderText) {
                loaderText.textContent = message;
            }
        }
    }

    hideLoader() {
        const loader = document.getElementById('moderation-loader');
        if (loader) {
            loader.style.display = 'none';
        }
    }
}

// Initialize moderation interface when DOM is ready
document.addEventListener('DOMContentLoaded', function() {
    if (document.getElementById('shoutout-moderation-container')) {
        window.castconductorShoutoutModeration = new CastConductorShoutoutModeration();
    }
});
