/**
 * Cast Conductor Proprietary License v5
 * SPDX-License-Identifier: LicenseRef-CastConductor-Proprietary-v5
 * Block Interaction Module: drag & resize logic for preview block wrapper.
 */
export function bindBlockInteractions(editor, wrap){
  if(!editor || !wrap) return;
  const snap = (v, step)=>{ const s = step || editor.canvasSnap || 8; return Math.round(v/s)*s; };
  const getBounds = ()=>{ const host = editor.previewContainer; return { w: host?.clientWidth||960, h: host?.clientHeight||Math.round((host?.clientWidth||960)*9/16) }; };
  const scale = editor.getPreviewScale?.() || { x:1, y:1 };
  let drag = null;
  const onMove = (e)=>{
    if(!drag) return; e.preventDefault();
    const dx = (e.clientX - drag.startX); const dy = (e.clientY - drag.startY);
    const step = editor.canvasSnap || 8; const bounds = getBounds();
    let x=drag.orig.x+dx, y=drag.orig.y+dy, w=drag.orig.w, h=drag.orig.h;
    if(drag.mode==='move'){
      x = snap(Math.max(0, Math.min(bounds.w - w, x)), step);
      y = snap(Math.max(0, Math.min(bounds.h - h, y)), step);
    } else {
      if (drag.dir.includes('r')) w = snap(Math.max(40, Math.min(bounds.w - drag.orig.x, drag.orig.w + dx)), step);
      if (drag.dir.includes('l')) { const nx = snap(Math.max(0, Math.min(drag.orig.x + drag.orig.w - 40, drag.orig.x + dx)), step); w = snap(drag.orig.w + (drag.orig.x - nx), step); x = nx; }
      if (drag.dir.includes('b')) h = snap(Math.max(40, Math.min(bounds.h - drag.orig.y, drag.orig.h + dy)), step);
      if (drag.dir.includes('t')) { const ny = snap(Math.max(0, Math.min(drag.orig.y + drag.orig.h - 40, drag.orig.y + dy)), step); h = snap(drag.orig.h + (drag.orig.y - ny), step); y = ny; }
    }
    wrap.style.left=`${x}px`; wrap.style.top=`${y}px`; wrap.style.width=`${w}px`; wrap.style.height=`${h}px`;
  };
  const onUp = ()=>{
    if(!drag) return;
    document.removeEventListener('mousemove', onMove); document.removeEventListener('mouseup', onUp);
    try {
      const x = parseInt(wrap.style.left,10)||0; const y = parseInt(wrap.style.top,10)||0;
      const w = parseInt(wrap.style.width,10)||wrap.offsetWidth||0; const h = parseInt(wrap.style.height,10)||wrap.offsetHeight||0;
      const lx = Math.round(x/(scale.x||1)); const ly = Math.round(y/(scale.y||1));
      const lw = Math.max(40, Math.round(w/(scale.x||1))); const lh = Math.max(40, Math.round(h/(scale.y||1)));
      if(!editor.currentConfig) editor.currentConfig = editor.getDefaultConfig();
      editor.currentConfig.layout = editor.currentConfig.layout || {};
      editor.currentConfig.layout.position = { x: lx, y: ly };
      editor.currentConfig.layout.width = lw; editor.currentConfig.layout.height = lh;
      editor.currentConfig.layout.x_position = lx; editor.currentConfig.layout.y_position = ly;
      editor.setControlValue?.('canvas-position-x', lx);
      editor.setControlValue?.('canvas-position-y', ly);
      editor.setControlValue?.('canvas-width', lw);
      editor.setControlValue?.('canvas-height', lh);
      editor._refreshLayoutPills?.();
      editor.markUnsaved?.();
      editor.triggerPreview?.();
    } catch(e){ console.warn('[block-interactions] persist failed', e); }
    drag=null;
  };
  wrap.addEventListener('mousedown', (e)=>{
    if (e.target.classList.contains('cc-block-handle')) return; // handles get their own listener
    drag = { mode:'move', startX:e.clientX, startY:e.clientY, orig:{ x:parseInt(wrap.style.left,10)||0, y:parseInt(wrap.style.top,10)||0, w:parseInt(wrap.style.width,10)||wrap.offsetWidth||0, h:parseInt(wrap.style.height,10)||wrap.offsetHeight||0 } };
    document.addEventListener('mousemove', onMove); document.addEventListener('mouseup', onUp);
  });
  ['tl','tr','bl','br'].forEach(dir=>{
    const handle = wrap.querySelector(`.cc-block-handle.${dir}`);
    if(!handle) return;
    handle.addEventListener('mousedown',(e)=>{
      e.preventDefault(); e.stopPropagation();
      drag = { mode:'resize', dir, startX:e.clientX, startY:e.clientY, orig:{ x:parseInt(wrap.style.left,10)||0, y:parseInt(wrap.style.top,10)||0, w:parseInt(wrap.style.width,10)||wrap.offsetWidth||0, h:parseInt(wrap.style.height,10)||wrap.offsetHeight||0 } };
      document.addEventListener('mousemove', onMove); document.addEventListener('mouseup', onUp);
    });
  });
}

if (typeof window !== 'undefined') { window.ccveBlockInteractions = { bind: bindBlockInteractions }; }