/*
 * CastConductor V5 – Container Presets
 * Defines canonical layout presets and a helper to seed them via existing REST endpoints.
 * All geometry is in 1280×720 authoring space (Roku's native resolution, upscales 1.5× for FHD)
 */
export function getContainerPresets() {
  return {
    standard_all_defaults: {
      label: 'Standard Broadcast Set',
      description: 'Full Screen, Thirds, Halves, Quarters – complete authoring baseline.',
      containers: [
        { name: 'Full Screen', position: 'full_screen', x_position: 0, y_position: 0, width: 1280, height: 720, z_index: 10 },
        { name: 'Upper Third', position: 'upper_third', x_position: 0, y_position: 0, width: 1280, height: 240, z_index: 90 },
        { name: 'Center Third', position: 'center_third', x_position: 0, y_position: 240, width: 1280, height: 240, z_index: 95 },
        { name: 'Lower Third', position: 'lower_third', x_position: 0, y_position: 480, width: 1280, height: 240, z_index: 100 },
        { name: 'Left Half', position: 'left_half', x_position: 0, y_position: 0, width: 640, height: 720, z_index: 20 },
        { name: 'Right Half', position: 'right_half', x_position: 640, y_position: 0, width: 640, height: 720, z_index: 20 },
        { name: 'Upper Left Quarter', position: 'upper_left_quarter', x_position: 0, y_position: 0, width: 640, height: 360, z_index: 30 },
        { name: 'Upper Right Quarter', position: 'upper_right_quarter', x_position: 640, y_position: 0, width: 640, height: 360, z_index: 30 },
        { name: 'Lower Left Quarter', position: 'lower_left_quarter', x_position: 0, y_position: 360, width: 640, height: 360, z_index: 30 },
        { name: 'Lower Right Quarter', position: 'lower_right_quarter', x_position: 640, y_position: 360, width: 640, height: 360, z_index: 30 },
        { name: 'Horizontal Upper Half', position: 'upper_half', x_position: 0, y_position: 0, width: 1280, height: 360, z_index: 25 },
        { name: 'Horizontal Lower Half', position: 'lower_half', x_position: 0, y_position: 360, width: 1280, height: 360, z_index: 25 },
        { name: 'Horizontal 1st Quarter', position: 'horizontal_first_quarter', x_position: 0, y_position: 0, width: 1280, height: 180, z_index: 35 },
        { name: 'Horizontal 2nd Quarter', position: 'horizontal_second_quarter', x_position: 0, y_position: 180, width: 1280, height: 180, z_index: 35 },
        { name: 'Horizontal 3rd Quarter', position: 'horizontal_third_quarter', x_position: 0, y_position: 360, width: 1280, height: 180, z_index: 35 },
        { name: 'Horizontal 4th Quarter', position: 'horizontal_fourth_quarter', x_position: 0, y_position: 540, width: 1280, height: 180, z_index: 35 }
      ]
    }
  };
}

/** Seed preset containers (append if not existing by position or name) */
export async function seedPresetContainers(editor, key, mode = 'append') {
  const presets = getContainerPresets();
  const preset = presets[key];
  if (!preset) throw new Error('Unknown preset');
  const existing = editor?._containersById || {};
  const existingNames = new Set(Object.values(existing).map(c => (c.name || '').toLowerCase()));
  const existingPositions = new Set(Object.values(existing).map(c => String(c.position || '').toLowerCase()));

  if (mode === 'replace') {
    Object.keys(existing).forEach(id => {
      try { fetch(`${castconductorCanvasAjax.rest_url}castconductor/v5/containers/${id}`, { method: 'DELETE', headers: { 'X-WP-Nonce': castconductorCanvasAjax.nonce } }).catch(()=>{}); } catch(_) {}
    });
  }

  let created = 0; let skipped = 0;
  for (const spec of preset.containers) {
    const posKey = String(spec.position || '').toLowerCase();
    const nameKey = String(spec.name || '').toLowerCase();
    if (existingPositions.has(posKey) || existingNames.has(nameKey)) { skipped++; continue; }
    const body = { ...spec, rotation_enabled: true, rotation_interval: 15, enabled: true };
    try {
      const resp = await fetch(`${castconductorCanvasAjax.rest_url}castconductor/v5/containers`, {
        method: 'POST',
        headers: { 'Content-Type': 'application/json', 'X-WP-Nonce': castconductorCanvasAjax.nonce },
        body: JSON.stringify(body)
      });
      if (resp.ok) { created++; }
    } catch (e) { console.warn('Preset create failed', spec, e); }
  }
  editor.showNotification(`Preset applied: ${created} created, ${skipped} skipped`, 'success');
  return { created, skipped };
}
