// Full Content Block Export Utilities (alpha)
// Provides: single block export, batch selected, all blocks

async function fetchBlock(id){
  try {
    // Prefer WP-provided REST base + nonce when available
    const base = (typeof castconductorCanvasAjax !== 'undefined' && castconductorCanvasAjax.rest_url)
      ? castconductorCanvasAjax.rest_url
      : '/wp-json/';
    const url = `${base}castconductor/v5/content-blocks/${id}`;
    const headers = (typeof castconductorCanvasAjax !== 'undefined' && castconductorCanvasAjax.nonce)
      ? { 'X-WP-Nonce': castconductorCanvasAjax.nonce }
      : {};
    const resp = await fetch(url, { headers });
    if(!resp.ok) return null;
    const data = await resp.json();
    // Support both wrapped ({ success, data }) and direct item responses
    if (data && typeof data === 'object' && 'data' in data) return data.data;
    return data || null;
  } catch(e){ console.warn('[CC] fetchBlock failed', id, e); return null; }
}

function sanitizeBlock(raw){
  if(!raw || typeof raw !== 'object') return null;
  const keep = {};
  // Copy over a curated subset
  const keys = [
    'id','name','type','enabled',
    // Optional legacy fields if present
    'title','slug','description','category',
    // Config blobs
    'visual_config','data_config',
    // Timestamps
    'created_at','updated_at'
  ];
  keys.forEach(k=>{ if(raw[k] !== undefined) keep[k]=raw[k]; });

  // Normalize and sanitize visual_config
  const vc = keep.visual_config = (keep.visual_config && typeof keep.visual_config === 'object') ? JSON.parse(JSON.stringify(keep.visual_config)) : {};

  // 1) Ensure layout.authored_space reflects 1280x720 authoring
  vc.layout = vc.layout || {};
  vc.layout.authored_space = 'admin_1280x720';

  // 2) Strip transient/rendered fields and legacy_* keys recursively
  const stripTransient = (obj)=>{
    if (!obj || typeof obj !== 'object') return obj;
    if (Array.isArray(obj)) return obj.map(stripTransient);
    const out = {};
    Object.keys(obj).forEach(k=>{
      if (k === 'renderedText' || k === 'renderedUrl' || k.startsWith('legacy_')) return; // drop
      const v = obj[k];
      out[k] = (typeof v === 'object' && v !== null) ? stripTransient(v) : v;
    });
    return out;
  };
  keep.visual_config = stripTransient(vc);

  // 3) Token layer normalization: ensure artwork token defaults to 240x240 minimum, square
  try {
    const layers = (keep.visual_config.token_layers || keep.visual_config.layers || []) || [];
    const normLayers = layers.map(l=>{
      if (!l || (l.kind !== 'token-text' && l.kind !== 'token-image')) return l;
      const copy = { ...l };
      // Remove transient fields if present
      delete copy.renderedText; delete copy.renderedUrl;
      if (copy.kind === 'token-image') {
        const tk = String(copy.token||'');
        if (/^track\.artwork$/i.test(tk)) {
          const w = parseInt(copy.width,10)||0, h = parseInt(copy.height,10)||0;
          const size = Math.max(240, w, h);
          copy.width = size; copy.height = size;
        }
      }
      return copy;
    });
    // Prefer token_layers if present; otherwise rewrite layers
    if (Array.isArray(keep.visual_config.token_layers) && keep.visual_config.token_layers.length) {
      keep.visual_config.token_layers = normLayers;
    } else if (Array.isArray(keep.visual_config.layers) && keep.visual_config.layers.length) {
      keep.visual_config.layers = normLayers;
    }
  } catch(_) {}

  // 4) Global artwork slot normalization (when used): ensure size at least 240x240 and square
  try {
    const aw = keep.visual_config.artwork || {};
    if (aw && aw.size) {
      const w = parseInt(aw.size.width,10)||0; const h = parseInt(aw.size.height,10)||0;
      const size = Math.max(240, w, h);
      keep.visual_config.artwork.size = { width: size, height: size };
    }
  } catch(_) {}

  return keep;
}

function renderExportModal(title, json){
  const html = `
  <style>
    .fcbexp-wrap{font:12px system-ui,Arial;display:flex;flex-direction:column;gap:8px;max-height:540px;}
    .fcbexp-code{width:100%;min-height:260px;font:11px/1.4 monospace;background:#0f172a;color:#e2e8f0;border:1px solid #334155;padding:6px;resize:vertical;}
    .fcbexp-actions{display:flex;justify-content:flex-end;gap:8px;}
    .fcbexp-btn{background:#2563eb;color:#fff;border:1px solid #1d4ed8;padding:4px 10px;border-radius:4px;cursor:pointer;font-size:12px;font-weight:600;}
    .fcbexp-close{background:#64748b;color:#fff;border:1px solid #475569;padding:4px 10px;border-radius:4px;cursor:pointer;font-size:12px;}
    .fcbexp-hint{font-size:11px;color:#94a3b8;}
  </style>
  <div class="fcbexp-wrap" id="ccve-full-block-export-modal">
    <div><strong>${title}</strong></div>
    <textarea readonly class="fcbexp-code" id="fcbexp-json">${json}</textarea>
    <div class="fcbexp-hint">Copy to replace activation wizard seeded defaults. Format: full content block object(s).</div>
    <div class="fcbexp-actions">
      <button type=button class="fcbexp-btn" id="fcbexp-download">Download JSON</button>
      <button type=button class="fcbexp-close">Close</button>
    </div>
  </div>`;
  return html;
}

export async function openFullContentBlockExportModal(editor){
  if(!editor?.showModal) return;
  const id = editor.currentContentBlockId;
  if(!id){ console.warn('[CC] no currentContentBlockId for full export'); return; }
  // Enforce save-first export to avoid unsaved diffs in clean DB workflows
  if (editor.unsavedChanges) {
    const msg = 'You have unsaved changes. Please save the content block before exporting.';
    if (editor.showNotification) editor.showNotification(msg, 'warn');
    else alert(msg);
    return;
  }
  const raw = await fetchBlock(id);
  const sanitized = sanitizeBlock(raw) || {};
  const payload = { export_format_version:1, generated_at:new Date().toISOString(), content_block: sanitized };
  const json = JSON.stringify(payload, null, 2);
  editor.showModal('Export Content Block', renderExportModal('Export Content Block', json));
  const root=document.getElementById('ccve-full-block-export-modal');
  if(root){
    // Close button
    root.querySelector('.fcbexp-close').addEventListener('click', ()=>editor.closeModal());
    // Download handler
    const dlBtn = root.querySelector('#fcbexp-download');
    if (dlBtn){
      dlBtn.addEventListener('click', ()=>{
        try{
          const ta = root.querySelector('#fcbexp-json');
          const text = ta ? ta.value : json;
          const blob = new Blob([text], { type: 'application/json' });
          const url = URL.createObjectURL(blob);
          // Build a sensible filename: content-block-{id}-{slugified-name}-{YYYYMMDD}.json
          const today = new Date();
          const y = today.getFullYear();
          const m = String(today.getMonth()+1).padStart(2,'0');
          const d = String(today.getDate()).padStart(2,'0');
          const parsed = (()=>{ try { return JSON.parse(text); } catch(_) { return null; } })() || {};
          const cb = parsed.content_block || {};
          const baseName = (cb.slug || cb.name || `block-${id}`).toString().toLowerCase().replace(/[^a-z0-9]+/g,'-').replace(/(^-|-$)/g,'') || `block-${id}`;
          const filename = `content-block-${cb.id||id}-${baseName}-${y}${m}${d}.json`;
          const a = document.createElement('a');
          a.href = url; a.download = filename;
          document.body.appendChild(a); a.click(); document.body.removeChild(a);
          URL.revokeObjectURL(url);
        }catch(err){ console.warn('[CC] export download failed', err); }
      });
    }
  }
}

export async function openFullBatchExportModal(editor){
  if(!editor?.showModal) return;
  // Enforce save-first gating like single export
  if (editor.unsavedChanges) {
    const msg = 'You have unsaved changes. Please save before generating a batch export.';
    if (editor.showNotification) editor.showNotification(msg, 'warn');
    else alert(msg);
    return;
  }
  // Reuse list from earlier caching approach if available
  const list = (window.ccveContentBlocksListCache||[]).slice(0,250);
  if(!list.length){ editor.showNotification?.('No cached content blocks list for batch export','warn'); }
  // Build simple selection UI dynamically first
  const selectionHtml = `
    <style>.fcbsel-wrap{font:12px system-ui,Arial;display:flex;flex-direction:column;gap:8px;max-height:480px;} .fcbsel-list{border:1px solid #334155;padding:6px;overflow:auto;max-height:240px;background:#0f172a;color:#e2e8f0} .fcbsel-list label{display:flex;align-items:center;gap:6px;padding:2px 0;font-size:11px;} .fcbsel-actions{display:flex;justify-content:space-between;margin-top:6px} .fcbsel-btn{background:#2563eb;color:#fff;border:1px solid #1d4ed8;padding:4px 10px;border-radius:4px;cursor:pointer;font-size:12px;font-weight:600;} .fcbsel-close{background:#64748b;color:#fff;border:1px solid #475569;padding:4px 10px;border-radius:4px;cursor:pointer;font-size:12px;} textarea.fcbsel-out{width:100%;min-height:200px;font:11px monospace;background:#0f172a;color:#e2e8f0;border:1px solid #334155;padding:6px}</style>
    <div class="fcbsel-wrap" id="ccve-full-batch-export-modal">
      <div><strong>Select Content Blocks (Full Export)</strong></div>
      <div class="fcbsel-list">${list.map(cb=>`<label><input type=checkbox value="${cb.id}"> ${cb.id} – ${cb.name||cb.title||'Block'}</label>`).join('')||'No cached list.'}</div>
      <button type=button class="fcbsel-btn" id="fcbsel-generate">Generate JSON</button>
      <textarea class="fcbsel-out" id="fcbsel-output" placeholder="Batch export JSON appears here" readonly></textarea>
      <div class="fcbsel-actions"><button type=button class="fcbsel-close">Close</button></div>
    </div>`;
  editor.showModal('Batch Export Content Blocks', selectionHtml);
  const root=document.getElementById('ccve-full-batch-export-modal'); if(!root) return;
  root.querySelector('.fcbsel-close').addEventListener('click', ()=>editor.closeModal());
  root.querySelector('#fcbsel-generate').addEventListener('click', async ()=>{
    const ids = Array.from(root.querySelectorAll('input[type=checkbox]:checked')).map(i=>parseInt(i.value,10)).filter(Boolean);
    if(!ids.length){ return; }
    const blocks=[];
    for(const id of ids){
      const raw = await fetchBlock(id); if(!raw) continue; const sanitized = sanitizeBlock(raw); if(sanitized) blocks.push(sanitized);
    }
    const json = JSON.stringify({ export_format_version:1, generated_at:new Date().toISOString(), content_blocks: blocks }, null, 2);
    root.querySelector('#fcbsel-output').value = json;
  });
}

export async function openFullAllBlocksExportModal(editor){
  if(!editor?.showModal) return;
  // Enforce save-first gating like single export
  if (editor.unsavedChanges) {
    const msg = 'You have unsaved changes. Please save before exporting all content blocks.';
    if (editor.showNotification) editor.showNotification(msg, 'warn');
    else alert(msg);
    return;
  }
  // Fetch list endpoint directly (safer than relying solely on cache)
  let list=[];
  try {
    const resp = await fetch('/wp-json/castconductor/v5/content-blocks');
    if(resp.ok){ const data = await resp.json(); list = (data?.data||[]).slice(0,500); }
  } catch(e){ console.warn('[CC] fetch all blocks list failed', e); }
  const blocks=[];
  for(const item of list){
    const raw = await fetchBlock(item.id); if(!raw) continue; const sanitized = sanitizeBlock(raw); if(sanitized) blocks.push(sanitized);
  }
  const json = JSON.stringify({ export_format_version:1, generated_at:new Date().toISOString(), content_blocks: blocks }, null, 2);
  editor.showModal('Export All Content Blocks', renderExportModal('Export All Content Blocks', json));
  const root=document.getElementById('ccve-full-block-export-modal'); if(root){ root.querySelector('.fcbexp-close').addEventListener('click', ()=>editor.closeModal()); }
}

export default { openFullContentBlockExportModal, openFullBatchExportModal, openFullAllBlocksExportModal };
