/**
 * Cast Conductor Proprietary License v5
 * SPDX-License-Identifier: LicenseRef-CastConductor-Proprietary-v5
 * Geometry HUD & Scale Toggle Module
 * Responsibilities:
 *  - Provide scale toggle button (fit vs 1x)
 *  - Provide geometry HUD showing logical vs final rect and scale
 *  - Expose updateHud(editor) helper
 */
export function initGeometryHud(editor){
  if (!editor || !editor.previewContainer) return;
  if (editor._geometryHudInit) return; editor._geometryHudInit = true;
  // Restore persisted scale mode (if any) before building UI
  try {
    const stored = window.localStorage.getItem('ccve.previewScaleMode');
    if (stored === 'fit' || stored === '1x') editor.previewScaleMode = stored;
  } catch(_) {}
  const pc = editor.previewContainer;
  // Scale toggle
  const btn = document.createElement('button');
  btn.type='button';
  btn.className='ccve-scale-toggle';
  btn.textContent = 'Scale: ' + editor.previewScaleMode;
  Object.assign(btn.style,{ position:'absolute', bottom:'6px', right:'6px', zIndex:60, fontSize:'11px', padding:'4px 8px', background:'#1e293b', color:'#fff', border:'1px solid #334155', borderRadius:'4px', cursor:'pointer', opacity:'0.85' });
  btn.addEventListener('click', ()=>{
    editor.previewScaleMode = (editor.previewScaleMode === 'fit') ? '1x' : 'fit';
    try { window.localStorage.setItem('ccve.previewScaleMode', editor.previewScaleMode); } catch(_) {}
    btn.textContent = 'Scale: ' + editor.previewScaleMode;
    try { editor.updateBlockEditorRectFromConfig(); } catch{}
    try { updateHud(editor); } catch{}
  });
  pc.appendChild(btn);
  // HUD
  const hud = document.createElement('div');
  hud.id='ccve-geometry-hud';
  Object.assign(hud.style,{ position:'absolute', left:'6px', bottom:'6px', background:'rgba(0,0,0,0.55)', color:'#fff', font:'10px/1.3 ui-monospace,monospace', padding:'6px 8px', borderRadius:'4px', maxWidth:'260px', zIndex:60, pointerEvents:'none', whiteSpace:'pre-wrap' });
  hud.textContent='Geometry HUD';
  pc.appendChild(hud);
}

export function updateHud(editor){
  if (!editor) return; const hud = document.getElementById('ccve-geometry-hud'); if(!hud) return;
  try {
    const l = editor.currentConfig?.layout || {};
    const logical = { x:l.x_position??l.x??0, y:l.y_position??l.y??0, w:l.width||0, h:l.height||0 };
    const scale = editor.getPreviewScale?.() || { x:1, y:1 };
    const wrap = editor.previewContainer?.querySelector('.cc-block-editor-wrap');
    const rect = wrap ? { w: wrap.offsetWidth, h: wrap.offsetHeight, x: wrap.offsetLeft, y: wrap.offsetTop } : { w:0,h:0,x:0,y:0 };
    hud.textContent = `Logical: ${logical.w}x${logical.h} @ ${logical.x},${logical.y}\nFinal:   ${rect.w}x${rect.h} @ ${rect.x},${rect.y}\nScale:   ${(scale.x||1).toFixed(3)}x mode=${editor.previewScaleMode}`;
  } catch(e) { hud.textContent='HUD error: '+e.message; }
}

if (typeof window !== 'undefined') {
  window.ccveGeometryHud = { init: initGeometryHud, update: (ed)=> updateHud(ed||window.castconductorCanvasEditor) };
}