/**
 * Layer Constants and Type Definitions
 * 
 * Defines all layer kinds and provides validation/factory functions
 * for the unified layer system.
 * 
 * @module layer-constants
 * @since 5.2.1 - Unified Layer System
 */

// ============================================================================
// LAYER KIND CONSTANTS
// ============================================================================

/**
 * Text layer kinds
 */
export const LAYER_KIND_TOKEN_TEXT = 'token-text';           // Dynamic text from metadata tokens
export const LAYER_KIND_STATIC_TEXT = 'static-text';         // User-typed custom text
export const LAYER_KIND_WP_POST_TITLE = 'wordpress-post-title';       // WordPress post title (future)
export const LAYER_KIND_WP_POST_CONTENT = 'wordpress-post-content';   // WordPress post body (future)
export const LAYER_KIND_WP_POST_EXCERPT = 'wordpress-post-excerpt';   // WordPress post excerpt (future)

/**
 * Image layer kinds
 */
export const LAYER_KIND_TOKEN_IMAGE = 'token-image';         // Dynamic images from tokens
export const LAYER_KIND_STATIC_IMAGE = 'static-image';       // Single uploaded/selected image
export const LAYER_KIND_SLIDESHOW_IMAGE = 'slideshow-image'; // Multiple images with transitions
export const LAYER_KIND_QR_IMAGE = 'qr-image';               // Generated QR codes
export const LAYER_KIND_BACKGROUND_IMAGE = 'background-image'; // Full-canvas backgrounds
export const LAYER_KIND_WP_POST_FEATURED = 'wordpress-post-featured'; // WordPress featured image (future)

/**
 * Video layer kinds
 */
export const LAYER_KIND_VIDEO = 'video';                     // Video as positioned layer

// ============================================================================
// UNIFIED CSS CLASS NAMING CONVENTION
// ============================================================================

/**
 * CSS Class constants for unified layer system
 * Use these instead of legacy ccve-* prefixed names
 */
export const CSS_CLASS = {
    // Layer container classes
    LAYER: 'unified-layer',                      // Base class for all layers
    LAYER_SELECTED: 'unified-layer--selected',   // Selected state
    LAYER_LOCKED: 'unified-layer--locked',       // Locked state
    LAYER_DRAGGING: 'unified-layer--dragging',   // Being dragged
    
    // Layer kind modifier (use with layer.kind)
    layerKind: (kind) => `unified-layer--${kind}`,
    
    // Internal content classes
    LAYER_TEXT_CONTENT: 'layer-text-content',    // Text content inside text layers
    LAYER_IMAGE_CONTENT: 'layer-image-content',  // Image inside image layers
    
    // Layer content wrappers
    TEXT_LAYER: 'layer-content text-layer',
    IMAGE_LAYER: 'layer-content image-layer',
};

/**
 * Data attribute constants
 */
export const DATA_ATTR = {
    LAYER_ID: 'data-layer-id',
    LAYER_KIND: 'data-layer-kind',
    TOKEN_IMAGE: 'data-token-image',
    TOKEN_TEXT: 'data-token-text',
};

/**
 * Interactive layer kinds (future)
 */
export const LAYER_KIND_MENU = 'menu-layer';                 // Navigable menu grids
export const LAYER_KIND_NAVIGATION = 'navigation-layer';     // Remote-controlled navigation
export const LAYER_KIND_FEED = 'feed-layer';                 // External feed aggregation

/**
 * Composite layer kinds
 */
export const LAYER_KIND_WP_POST = 'wordpress-post';          // WordPress post with multiple fields

/**
 * All text layer kinds
 */
export const TEXT_LAYER_KINDS = [
    LAYER_KIND_TOKEN_TEXT,
    LAYER_KIND_STATIC_TEXT,
    LAYER_KIND_WP_POST_TITLE,
    LAYER_KIND_WP_POST_CONTENT,
    LAYER_KIND_WP_POST_EXCERPT
];

/**
 * All image layer kinds
 */
export const IMAGE_LAYER_KINDS = [
    LAYER_KIND_TOKEN_IMAGE,
    LAYER_KIND_STATIC_IMAGE,
    LAYER_KIND_SLIDESHOW_IMAGE,
    LAYER_KIND_QR_IMAGE,
    LAYER_KIND_BACKGROUND_IMAGE,
    LAYER_KIND_WP_POST_FEATURED
];

/**
 * All video layer kinds
 */
export const VIDEO_LAYER_KINDS = [
    LAYER_KIND_VIDEO
];

/**
 * All interactive layer kinds
 */
export const INTERACTIVE_LAYER_KINDS = [
    LAYER_KIND_MENU,
    LAYER_KIND_NAVIGATION,
    LAYER_KIND_FEED
];

/**
 * All layer kinds
 */
export const ALL_LAYER_KINDS = [
    ...TEXT_LAYER_KINDS,
    ...IMAGE_LAYER_KINDS,
    ...VIDEO_LAYER_KINDS,
    ...INTERACTIVE_LAYER_KINDS,
    LAYER_KIND_WP_POST
];

// ============================================================================
// LAYER PROPERTY DEFAULTS
// ============================================================================

/**
 * Universal properties that ALL layers must have
 */
export const UNIVERSAL_LAYER_PROPERTIES = {
    id: '',                    // Unique identifier (generated)
    kind: '',                  // Layer type from constants
    x: 100,                    // X position in pixels
    y: 100,                    // Y position in pixels
    width: 200,                // Width in pixels
    height: 100,               // Height in pixels
    visible: true,             // Show/hide toggle
    lock: false,               // Lock position/size
    zIndex: 10                 // Stacking order
};

/**
 * Default properties for text layers
 */
export const TEXT_LAYER_DEFAULTS = {
    text: '',                  // Text content (for static-text)
    token: '',                 // Metadata token (for token-text)
    font_family: 'Arial',
    font_size: 24,
    font_color: '#ffffff',
    font_weight: 'normal',     // normal, bold, 100-900
    font_style: 'normal',      // normal, italic
    text_align: 'left',        // left, center, right
    text_decoration: 'none',   // none, underline, line-through
    line_height: 1.2,
    letter_spacing: 0,
    text_shadow: 'none',
    max_lines: null,           // Text truncation (null = no limit)
    word_wrap: true
};

/**
 * Default properties for image layers
 */
export const IMAGE_LAYER_DEFAULTS = {
    url: '',                   // Image URL (for static-image)
    token: '',                 // Metadata token (for token-image)
    fallback_url: '',          // Fallback if token empty
    object_fit: 'cover',       // cover, contain, fill, none, scale-down
    opacity: 1.0,              // 0.0 to 1.0
    border_radius: 0,          // Rounded corners in pixels
    filter: 'none'             // CSS filter effects
};

/**
 * Default properties for slideshow layers
 */
export const SLIDESHOW_LAYER_DEFAULTS = {
    images: [],                // Array of {url, duration} objects
    transition: 'fade',        // fade, slide, zoom, none
    transition_duration: 500,  // Transition time in ms
    autoplay: true,
    loop: true,
    current_index: 0
};

/**
 * Default properties for QR layers
 */
export const QR_LAYER_DEFAULTS = {
    qr_data: '',               // Data to encode (usually URL)
    qr_size: 256,              // QR code size in pixels
    qr_color: '#000000',       // Foreground color
    qr_background: '#ffffff',  // Background color
    error_correction: 'M'      // L, M, Q, H
};

/**
 * Default properties for WordPress post layers
 */
export const WP_POST_LAYER_DEFAULTS = {
    post_id: null,             // Selected WordPress post ID
    fields: {
        title: {
            enabled: true,
            x: 0,
            y: 0,
            font_size: 32,
            font_color: '#ffffff'
        },
        content: {
            enabled: true,
            x: 0,
            y: 50,
            font_size: 18,
            font_color: '#cccccc',
            max_lines: 3
        },
        featured_image: {
            enabled: true,
            x: 0,
            y: 0,
            width: 200,
            height: 200,
            object_fit: 'cover'
        }
    }
};

/**
 * Default properties for video layers
 * Video layers play video content positioned within content blocks
 */
export const VIDEO_LAYER_DEFAULTS = {
    url: '',                   // Video URL (HLS, MP4, etc.)
    format: 'auto',            // auto, hls, mp4, dash - auto-detect from URL
    poster: '',                // Poster/thumbnail image URL
    autoplay: true,            // Start playing automatically
    loop: true,                // Loop when video ends
    muted: false,              // Mute audio (true for background-style videos)
    controls: false,           // Show playback controls (Roku doesn't support in-layer)
    object_fit: 'contain',     // contain, cover - how video fits in bounds
    fallback_image: '',        // Fallback image if video fails to load
    // Deep Linking (Phase 6)
    content_id: '',            // Unique content ID for deep linking (auto-generated if empty)
    media_type: 'shortFormVideo', // Roku media type: movie, episode, shortFormVideo, series, live
    searchable: true,          // Include in Roku Feed export for Universal Search
    title: '',                 // Override title for Roku Feed (uses content block title if empty)
    description: ''            // Override description for Roku Feed
};

/**
 * Default properties for feed layers
 * Feed layers display content from RSS, MRSS, Atom, JSON, or API feeds
 */
export const FEED_LAYER_DEFAULTS = {
    // Position and size defaults for podcast/feed content blocks
    x: 360,                    // Default X position
    y: 160,                    // Default Y position
    width: 600,                // Default width
    height: 400,               // Default height
    
    // Feed configuration
    feed_url: '',              // Feed URL to fetch
    feed_type: 'auto',         // auto, rss, mrss, atom, json, roku, youtube, soundcloud
    api_key: '',               // API key for YouTube/SoundCloud (optional)
    max_items: 300,            // Maximum items to display
    layout: 'grid',            // grid, list, carousel, featured
    columns: 3,                // Number of columns (for grid layout)
    item_spacing: 150,         // Spacing between items
    
    // Thumbnail settings
    show_thumbnail: true,      // Show item thumbnails
    thumbnail_width: 200,      // Thumbnail width in pixels
    thumbnail_height: 200,     // Thumbnail height (square for podcast art)
    
    // Show/Hide elements
    show_title: true,          // Show item titles
    show_description: true,    // Show item descriptions
    show_duration: false,      // Hide duration by default
    show_published: true,      // Show published date
    
    // Typography
    title_font_size: 14,       // Title font size
    title_font_color: '#ffffff',// Title color
    title_max_lines: 2,        // Maximum lines for title
    description_font_size: 12, // Description font size
    description_font_color: '#cccccc', // Description color
    description_max_lines: 2,  // Maximum lines for description
    
    // Focus and Selection
    selection_style: 'highlight', // highlight, border, scale
    selection_color: '#0077ff', // Selection indicator color
    focus_scale: 1.2,          // Focus zoom scale (1.2 = 20% larger)
    
    // Container and Layout
    show_container: false,     // Hide container background by default
    center_focus_mode: true,   // Center focus for carousel mode
    
    // Audio controls
    show_progress_bar: true,   // Show audio progress bar
    
    // Transport Overlay Settings
    enable_transport_overlay: true,  // Per-menu toggle for mini player bar
    transport_trigger_key: 'transport',  // 'transport' (any), 'replay', 'play'
    
    // Transport Overlay Styling (mini player bar)
    transport_bg_color: '#000000',
    transport_bg_opacity: 0.95,
    transport_title_color: '#ffffff',
    transport_progress_bg_color: '#333333',
    transport_progress_fill_color: '#00aa00',
    transport_time_color: '#aaaaaa',
    transport_button_color: '#444444',
    transport_button_focus_color: '#0077ff',
    transport_play_button_color: '#00aa00',
    transport_now_playing_color: '#0077ff',
    transport_text_color: '#ffffff',
    
    // Refresh settings
    auto_refresh: false,       // Auto-refresh feed
    refresh_interval: 300,     // Refresh interval in seconds (5 min default)
    on_select_action: 'play',  // play, details, navigate
    cached_items: []           // Cached feed items (populated by parser)
};

/**
 * Default animation properties (optional)
 */
export const ANIMATION_DEFAULTS = {
    enabled: false,
    type: 'fade_in',           // fade_in, slide_in, zoom_in, etc.
    duration: 500,             // Animation duration in ms
    easing: 'ease-in-out',     // ease, ease-in, ease-out, ease-in-out, linear
    delay: 0                   // Delay before animation starts
};

/**
 * Default ticker properties (optional)
 */
export const TICKER_DEFAULTS = {
    enabled: false,
    speed: 50,                 // Pixels per second
    direction: 'left',         // left, right, up, down
    pause_on_hover: false
};

// ============================================================================
// VALIDATION FUNCTIONS
// ============================================================================

/**
 * Validate that a layer has all required universal properties
 * 
 * @param {Object} layer - Layer object to validate
 * @returns {Object} Validation result {valid: boolean, errors: string[]}
 */
export function validateUniversalProperties(layer) {
    const errors = [];
    
    if (!layer.id || typeof layer.id !== 'string') {
        errors.push('Layer must have a valid id');
    }
    
    if (!layer.kind || !ALL_LAYER_KINDS.includes(layer.kind)) {
        errors.push(`Invalid layer kind: ${layer.kind}`);
    }
    
    if (typeof layer.x !== 'number') errors.push('Layer x must be a number');
    if (typeof layer.y !== 'number') errors.push('Layer y must be a number');
    if (typeof layer.width !== 'number') errors.push('Layer width must be a number');
    if (typeof layer.height !== 'number') errors.push('Layer height must be a number');
    if (typeof layer.visible !== 'boolean') errors.push('Layer visible must be a boolean');
    if (typeof layer.lock !== 'boolean') errors.push('Layer lock must be a boolean');
    if (typeof layer.zIndex !== 'number') errors.push('Layer zIndex must be a number');
    
    return {
        valid: errors.length === 0,
        errors
    };
}

/**
 * Validate layer-kind-specific properties
 * 
 * @param {Object} layer - Layer object to validate
 * @returns {Object} Validation result {valid: boolean, errors: string[]}
 */
export function validateKindSpecificProperties(layer) {
    const errors = [];
    
    switch (layer.kind) {
        case LAYER_KIND_TOKEN_TEXT:
            if (!layer.token) errors.push('token-text layer must have a token');
            break;
            
        case LAYER_KIND_STATIC_TEXT:
            if (typeof layer.text !== 'string') errors.push('static-text layer must have text');
            break;
            
        case LAYER_KIND_TOKEN_IMAGE:
            if (!layer.token) errors.push('token-image layer must have a token');
            break;
            
        case LAYER_KIND_STATIC_IMAGE:
            if (!layer.url) errors.push('static-image layer must have a url');
            break;
            
        case LAYER_KIND_SLIDESHOW_IMAGE:
            if (!Array.isArray(layer.images)) {
                errors.push('slideshow-image layer must have images array');
            } else if (layer.images.length === 0) {
                errors.push('slideshow-image layer must have at least one image');
            }
            break;
            
        case LAYER_KIND_QR_IMAGE:
            if (!layer.qr_data) errors.push('qr-image layer must have qr_data');
            break;
            
        case LAYER_KIND_WP_POST:
            if (!layer.post_id) errors.push('wordpress-post layer must have post_id');
            if (!layer.fields) errors.push('wordpress-post layer must have fields object');
            break;
            
        case LAYER_KIND_VIDEO:
            if (!layer.url) errors.push('video layer must have a url');
            break;
    }
    
    return {
        valid: errors.length === 0,
        errors
    };
}

/**
 * Validate complete layer object
 * 
 * @param {Object} layer - Layer object to validate
 * @returns {Object} Validation result {valid: boolean, errors: string[]}
 */
export function validateLayer(layer) {
    const universalValidation = validateUniversalProperties(layer);
    const kindValidation = validateKindSpecificProperties(layer);
    
    return {
        valid: universalValidation.valid && kindValidation.valid,
        errors: [...universalValidation.errors, ...kindValidation.errors]
    };
}

// ============================================================================
// LAYER FACTORY FUNCTIONS
// ============================================================================

/**
 * Generate unique layer ID
 * 
 * @returns {string} Unique layer ID
 */
export function generateLayerId() {
    return `layer_${Date.now()}_${Math.random().toString(36).substr(2, 9)}`;
}

/**
 * Create a new layer with universal properties and kind-specific defaults
 * 
 * @param {string} kind - Layer kind from constants
 * @param {Object} overrides - Properties to override defaults
 * @returns {Object} New layer object
 */
export function createLayer(kind, overrides = {}) {
    // Start with universal properties
    const layer = {
        ...UNIVERSAL_LAYER_PROPERTIES,
        id: generateLayerId(),
        kind
    };
    
    // Add kind-specific defaults
    if (TEXT_LAYER_KINDS.includes(kind)) {
        Object.assign(layer, TEXT_LAYER_DEFAULTS);
    }
    
    if (IMAGE_LAYER_KINDS.includes(kind)) {
        Object.assign(layer, IMAGE_LAYER_DEFAULTS);
    }
    
    if (kind === LAYER_KIND_SLIDESHOW_IMAGE) {
        Object.assign(layer, SLIDESHOW_LAYER_DEFAULTS);
    }
    
    if (kind === LAYER_KIND_QR_IMAGE) {
        Object.assign(layer, QR_LAYER_DEFAULTS);
    }
    
    if (kind === LAYER_KIND_WP_POST) {
        Object.assign(layer, WP_POST_LAYER_DEFAULTS);
    }
    
    if (kind === LAYER_KIND_VIDEO) {
        Object.assign(layer, VIDEO_LAYER_DEFAULTS);
    }
    
    if (kind === LAYER_KIND_FEED) {
        Object.assign(layer, FEED_LAYER_DEFAULTS);
    }
    
    // Apply overrides
    Object.assign(layer, overrides);
    
    return layer;
}

/**
 * Create token text layer
 * 
 * @param {string} token - Metadata token (e.g., '{{track.title}}')
 * @param {Object} overrides - Additional properties
 * @returns {Object} New token text layer
 */
export function createTokenTextLayer(token, overrides = {}) {
    return createLayer(LAYER_KIND_TOKEN_TEXT, {
        token,
        ...overrides
    });
}

/**
 * Create static text layer
 * 
 * @param {string} text - Text content
 * @param {Object} overrides - Additional properties
 * @returns {Object} New static text layer
 */
export function createStaticTextLayer(text, overrides = {}) {
    return createLayer(LAYER_KIND_STATIC_TEXT, {
        text,
        ...overrides
    });
}

/**
 * Create token image layer
 * 
 * @param {string} token - Metadata token (e.g., 'track.artwork') - no curly braces for image tokens
 * @param {Object} overrides - Additional properties
 * @returns {Object} New token image layer
 */
export function createTokenImageLayer(token, overrides = {}) {
    return createLayer(LAYER_KIND_TOKEN_IMAGE, {
        token,
        ...overrides
    });
}

/**
 * Create static image layer
 * 
 * @param {string} url - Image URL
 * @param {Object} overrides - Additional properties
 * @returns {Object} New static image layer
 */
export function createStaticImageLayer(url, overrides = {}) {
    return createLayer(LAYER_KIND_STATIC_IMAGE, {
        url,
        ...overrides
    });
}

/**
 * Create slideshow image layer
 * 
 * @param {Array} images - Array of {url, duration} objects
 * @param {Object} overrides - Additional properties
 * @returns {Object} New slideshow image layer
 */
export function createSlideshowImageLayer(images = [], overrides = {}) {
    return createLayer(LAYER_KIND_SLIDESHOW_IMAGE, {
        images,
        width: 1920,  // Default to full canvas width
        height: 1080, // Default to full canvas height
        x: 0,
        y: 0,
        zIndex: 0,    // Default to background
        ...overrides
    });
}

/**
 * Create QR image layer
 * 
 * @param {string} qrData - Data to encode (URL, text, etc.)
 * @param {Object} overrides - Additional properties
 * @returns {Object} New QR image layer
 */
export function createQrImageLayer(qrData, overrides = {}) {
    return createLayer(LAYER_KIND_QR_IMAGE, {
        qr_data: qrData,
        ...overrides
    });
}

/**
 * Create WordPress post layer
 * 
 * Composite layer that displays WordPress post content with configurable fields.
 * Each field (title, content, excerpt, featured_image) can be positioned independently.
 * 
 * @param {number} postId - WordPress post ID
 * @param {Object} overrides - Additional properties
 * @returns {Object} New WordPress post layer
 */
export function createWordPressPostLayer(postId, overrides = {}) {
    return createLayer(LAYER_KIND_WP_POST, {
        post_id: postId,
        // Field configuration with positioning and styling
        fields: {
            title: {
                enabled: true,
                x: 60,
                y: 60,
                width: 800,
                height: 80,
                font_family: 'system-ui, Arial, Helvetica, sans-serif',
                font_size: 48,
                font_weight: '700',
                color: '#ffffff',
                text_align: 'left',
                line_height: 1.2,
                text_shadow: { x: 2, y: 2, blur: 4, color: 'rgba(0,0,0,0.5)' }
            },
            content: {
                enabled: false,  // Usually too long for overlays
                x: 60,
                y: 160,
                width: 800,
                height: 400,
                font_family: 'system-ui, Arial, Helvetica, sans-serif',
                font_size: 24,
                font_weight: '400',
                color: '#ffffff',
                text_align: 'left',
                line_height: 1.5,
                max_length: 500,  // Character limit
                text_shadow: { x: 1, y: 1, blur: 2, color: 'rgba(0,0,0,0.5)' }
            },
            excerpt: {
                enabled: true,
                x: 60,
                y: 160,
                width: 800,
                height: 120,
                font_family: 'system-ui, Arial, Helvetica, sans-serif',
                font_size: 28,
                font_weight: '400',
                color: '#e0e0e0',
                text_align: 'left',
                line_height: 1.4,
                max_length: 200,
                text_shadow: { x: 1, y: 1, blur: 2, color: 'rgba(0,0,0,0.5)' }
            },
            featured_image: {
                enabled: true,
                x: 960,
                y: 60,
                width: 800,
                height: 600,
                fit: 'cover',
                border_radius: 8
            }
        },
        // Layout preset (can be 'left', 'right', 'top', 'bottom', 'custom')
        layout_preset: 'left',
        // Caching
        _cached_post: null,       // Cached post data to avoid repeated API calls
        _cache_timestamp: null,   // When cache was last updated
        ...overrides
    });
}

/**
 * Create video layer
 * 
 * Video layer plays video content positioned within content blocks.
 * Supports HLS, MP4, DASH formats. Video plays at specified position/size
 * with configurable autoplay, loop, and mute options.
 * 
 * @param {string} url - Video URL (HLS .m3u8, MP4, etc.)
 * @param {Object} overrides - Additional properties
 * @returns {Object} New video layer
 */
export function createVideoLayer(url, overrides = {}) {
    // Auto-detect format from URL
    let format = 'auto';
    if (url) {
        if (url.includes('.m3u8')) format = 'hls';
        else if (url.includes('.mp4')) format = 'mp4';
        else if (url.includes('.mpd')) format = 'dash';
    }
    
    return createLayer(LAYER_KIND_VIDEO, {
        url,
        format,
        width: 400,     // Default size - 16:9 ratio
        height: 225,
        ...overrides
    });
}

/**
 * Create a feed layer
 * 
 * Displays content from external feeds (RSS, MRSS, Atom, JSON, YouTube, etc.)
 * in grid, list, or carousel layouts with navigation support.
 * 
 * @param {string} feedUrl - Feed URL to parse
 * @param {Object} overrides - Additional properties
 * @returns {Object} New feed layer
 */
export function createFeedLayer(feedUrl, overrides = {}) {
    // Auto-detect feed type from URL
    let feedType = 'auto';
    if (feedUrl) {
        if (feedUrl.includes('youtube.com') || feedUrl.includes('youtu.be')) {
            feedType = 'youtube';
        } else if (feedUrl.includes('soundcloud.com')) {
            feedType = 'soundcloud';
        } else if (feedUrl.endsWith('.json')) {
            feedType = 'json';
        } else if (feedUrl.endsWith('.atom')) {
            feedType = 'atom';
        }
        // Default to 'auto' for RSS/MRSS which will be detected by parser
    }
    
    return createLayer(LAYER_KIND_FEED, {
        feed_url: feedUrl,
        feed_type: feedType,
        width: 600,     // Default size for grid layout
        height: 400,
        ...overrides
    });
}

// ============================================================================
// LAYER UTILITY FUNCTIONS
// ============================================================================

/**
 * Check if layer is a text layer
 * 
 * @param {Object} layer - Layer to check
 * @returns {boolean} True if text layer
 */
export function isTextLayer(layer) {
    return TEXT_LAYER_KINDS.includes(layer.kind);
}

/**
 * Check if layer is an image layer
 * 
 * @param {Object} layer - Layer to check
 * @returns {boolean} True if image layer
 */
export function isImageLayer(layer) {
    return IMAGE_LAYER_KINDS.includes(layer.kind);
}

/**
 * Check if layer is an interactive layer
 * 
 * @param {Object} layer - Layer to check
 * @returns {boolean} True if interactive layer
 */
export function isInteractiveLayer(layer) {
    return INTERACTIVE_LAYER_KINDS.includes(layer.kind);
}

/**
 * Check if layer is a video layer
 * 
 * @param {Object} layer - Layer to check
 * @returns {boolean} True if video layer
 */
export function isVideoLayer(layer) {
    return VIDEO_LAYER_KINDS.includes(layer.kind);
}

/**
 * Check if layer is a feed layer
 * 
 * @param {Object} layer - Layer to check
 * @returns {boolean} True if feed layer
 */
export function isFeedLayer(layer) {
    return layer.kind === LAYER_KIND_FEED;
}

/**
 * Get human-readable layer type name
 * 
 * @param {string} kind - Layer kind constant
 * @returns {string} Human-readable name
 */
export function getLayerTypeName(kind) {
    const names = {
        [LAYER_KIND_TOKEN_TEXT]: 'Token Text',
        [LAYER_KIND_STATIC_TEXT]: 'Static Text',
        [LAYER_KIND_TOKEN_IMAGE]: 'Token Image',
        [LAYER_KIND_STATIC_IMAGE]: 'Static Image',
        [LAYER_KIND_SLIDESHOW_IMAGE]: 'Slideshow',
        [LAYER_KIND_QR_IMAGE]: 'QR Code',
        [LAYER_KIND_BACKGROUND_IMAGE]: 'Background',
        [LAYER_KIND_VIDEO]: 'Video',
        [LAYER_KIND_WP_POST]: 'WordPress Post',
        [LAYER_KIND_WP_POST_TITLE]: 'Post Title',
        [LAYER_KIND_WP_POST_CONTENT]: 'Post Content',
        [LAYER_KIND_WP_POST_FEATURED]: 'Featured Image',
        [LAYER_KIND_MENU]: 'Menu',
        [LAYER_KIND_NAVIGATION]: 'Navigation',
        [LAYER_KIND_FEED]: 'Feed'
    };
    
    return names[kind] || kind;
}

/**
 * Clone layer with new ID
 * 
 * @param {Object} layer - Layer to clone
 * @returns {Object} Cloned layer with new ID
 */
export function cloneLayer(layer) {
    return {
        ...layer,
        id: generateLayerId()
    };
}
