/**
 * Cast Conductor Proprietary License v5
 * SPDX-License-Identifier: LicenseRef-CastConductor-Proprietary-v5
 * 
 * Layer Icons and Labels Module
 * 
 * Purpose:
 * Centralized mapping of layer kinds to display icons and human-readable labels
 * for consistent UI presentation across Layer Manager, Add Layer dropdown, etc.
 * 
 * Usage:
 * import { getLayerIcon, getLayerLabel, getLayerDescription } from './layer-icons.js';
 * 
 * const icon = getLayerIcon('token-text'); // Returns '🔤'
 * const label = getLayerLabel('slideshow-image'); // Returns 'Slideshow'
 */

/**
 * Layer Kind Icon Mapping
 * 
 * All layer kinds map to visual icons (emoji or SVG) for quick recognition
 * Unified system supports 8 core layer types + legacy fallbacks
 */
const LAYER_ICONS = {
    // Text Layers
    'token-text': '🔤',      // Dynamic text from metadata tokens
    'static-text': '✏️',     // Custom editable text
    
    // Image Layers
    'token-image': '🎨',     // Dynamic images from tokens (artwork, logos)
    'static-image': '🖼️',    // Uploaded or media library images
    'slideshow-image': '🎞️', // Multiple images with transitions
    'background-image': '🌄', // Full-canvas background
    
    // Special Layers
    'qr-image': 'QR',        // QR code generator (will use SVG in UI)
    'wordpress-post': '📰',  // WordPress post content (Phase 2)
    
    // Legacy/Background Layers (hydrated from old config)
    'background': '🖌️',      // Legacy background pseudo-layer
    'overlay': '◪',         // Background overlay layer
    'image': '🖼️',          // Background image layer
    'gradient': '🌈',        // Background gradient layer
};

/**
 * Layer Kind Label Mapping
 * 
 * Human-readable names for layer kinds
 * Used in Layer Manager list, properties panel, etc.
 */
const LAYER_LABELS = {
    // Text Layers
    'token-text': 'Token Text',
    'static-text': 'Static Text',
    
    // Image Layers
    'token-image': 'Token Image',
    'static-image': 'Static Image',
    'slideshow-image': 'Slideshow',
    'background-image': 'Background Image',
    
    // Special Layers
    'qr-image': 'QR Code',
    'wordpress-post': 'WordPress Post',
    
    // Legacy/Background Layers
    'background': 'Background',
    'overlay': 'Overlay',
    'image': 'Background Image',
    'gradient': 'Gradient',
};

/**
 * Layer Kind Description Mapping
 * 
 * Short descriptions explaining layer purpose
 * Used in Add Layer dropdown, tooltips, help text
 */
const LAYER_DESCRIPTIONS = {
    // Text Layers
    'token-text': 'Dynamic text from metadata tokens',
    'static-text': 'Custom editable text',
    
    // Image Layers
    'token-image': 'Dynamic images from tokens (artwork, logos)',
    'static-image': 'Upload or select from media library',
    'slideshow-image': 'Multiple images with transitions',
    'background-image': 'Full-canvas background',
    
    // Special Layers
    'qr-image': 'Generate scannable QR codes',
    'wordpress-post': 'Link to WordPress post content',
    
    // Legacy/Background Layers
    'background': 'Legacy background configuration',
    'overlay': 'Color overlay with transparency',
    'image': 'Background image layer',
    'gradient': 'Gradient background layer',
};

/**
 * Get icon for a layer kind
 * 
 * @param {string} kind - Layer kind (e.g., 'token-text', 'slideshow-image')
 * @returns {string} - Icon emoji or text (e.g., '🔤', 'QR')
 */
export function getLayerIcon(kind) {
    return LAYER_ICONS[kind] || '📦'; // Default fallback: package icon
}

/**
 * Get label for a layer kind
 * 
 * @param {string} kind - Layer kind (e.g., 'token-text', 'slideshow-image')
 * @returns {string} - Human-readable label (e.g., 'Token Text', 'Slideshow')
 */
export function getLayerLabel(kind) {
    return LAYER_LABELS[kind] || 'Unknown Layer';
}

/**
 * Get description for a layer kind
 * 
 * @param {string} kind - Layer kind (e.g., 'token-text', 'slideshow-image')
 * @returns {string} - Short description (e.g., 'Dynamic text from metadata tokens')
 */
export function getLayerDescription(kind) {
    return LAYER_DESCRIPTIONS[kind] || 'Custom layer';
}

/**
 * Get full display name for a layer
 * 
 * Combines icon + label for rich display
 * Example: "🔤 Token Text"
 * 
 * @param {string} kind - Layer kind
 * @returns {string} - Formatted display name
 */
export function getLayerDisplayName(kind) {
    const icon = getLayerIcon(kind);
    const label = getLayerLabel(kind);
    return `${icon} ${label}`;
}

/**
 * Get QR Code icon as SVG
 * 
 * Returns inline SVG for QR code icon
 * Used instead of emoji for better visual quality
 * 
 * @param {Object} options - SVG options
 * @param {string} options.size - Size (e.g., '20px')
 * @param {string} options.color - Fill color (default: currentColor)
 * @returns {string} - SVG markup
 */
export function getQRIconSVG({ size = '20px', color = 'currentColor' } = {}) {
    return `
        <svg style="width: ${size}; height: ${size};" fill="${color}" viewBox="0 0 20 20">
            <path fill-rule="evenodd" d="M4 3a2 2 0 00-2 2v10a2 2 0 002 2h12a2 2 0 002-2V5a2 2 0 00-2-2H4zm3 2h6v4H7V5zm8 8v2h1v-2h-1zm-2-2H7v4h6v-4zm2 0h1V9h-1v2zm1-4V5h-1v2h1zM5 9H4v2h1V9zm0 4H4v2h1v-2zM5 5H4v2h1V5z" clip-rule="evenodd"/>
        </svg>
    `.trim();
}

/**
 * Check if a layer kind is a text layer
 * 
 * @param {string} kind - Layer kind
 * @returns {boolean}
 */
export function isTextLayer(kind) {
    return kind === 'token-text' || kind === 'static-text';
}

/**
 * Check if a layer kind is an image layer
 * 
 * @param {string} kind - Layer kind
 * @returns {boolean}
 */
export function isImageLayer(kind) {
    return kind === 'token-image' || 
           kind === 'static-image' || 
           kind === 'slideshow-image' || 
           kind === 'background-image' ||
           kind === 'qr-image';
}

/**
 * Check if a layer kind is a background layer
 * 
 * @param {string} kind - Layer kind
 * @returns {boolean}
 */
export function isBackgroundLayer(kind) {
    return kind === 'background' || 
           kind === 'overlay' || 
           kind === 'image' || 
           kind === 'gradient' ||
           kind === 'background-image';
}

/**
 * Check if a layer kind is a legacy layer
 * 
 * Legacy layers are hydrated from old config schema
 * (pre-unified layer system)
 * 
 * @param {string} kind - Layer kind
 * @returns {boolean}
 */
export function isLegacyLayer(kind) {
    return kind === 'background' || 
           kind === 'overlay' || 
           kind === 'image' || 
           kind === 'gradient';
}

/**
 * Get all supported layer kinds
 * 
 * @returns {Array<string>} - Array of layer kind strings
 */
export function getAllLayerKinds() {
    return Object.keys(LAYER_ICONS);
}

/**
 * Get layer kinds by category
 * 
 * @param {string} category - 'text', 'image', 'special', 'background'
 * @returns {Array<string>} - Array of layer kind strings in category
 */
export function getLayerKindsByCategory(category) {
    const categories = {
        text: ['token-text', 'static-text'],
        image: ['token-image', 'static-image', 'slideshow-image', 'background-image'],
        special: ['qr-image', 'wordpress-post'],
        background: ['background', 'overlay', 'image', 'gradient'],
    };
    return categories[category] || [];
}

/**
 * Validate layer kind
 * 
 * @param {string} kind - Layer kind to validate
 * @returns {boolean} - True if kind is valid/supported
 */
export function isValidLayerKind(kind) {
    return kind in LAYER_ICONS;
}

/**
 * Get layer kind from legacy config section
 * 
 * Helper to map legacy config sections to unified layer kinds
 * 
 * @param {string} section - Legacy config section (e.g., 'typography', 'artwork')
 * @returns {string|null} - Corresponding layer kind or null
 */
export function getLegacyKind(section) {
    const legacyMap = {
        'typography': 'static-text',
        'artwork': 'static-image',
        'background': 'background',
    };
    return legacyMap[section] || null;
}
