/**
 * Cast Conductor Proprietary License v5
 * SPDX-License-Identifier: LicenseRef-CastConductor-Proprietary-v5
 * 
 * Copyright (c) 2025 CastConductor.com. All Rights Reserved.
 * 
 * This file is part of Cast Conductor ("Software"). The Software and its source
 * code constitute proprietary, confidential, and trade secret information of
 * CastConductor.com ("Company"). Any access or use is governed strictly by the
 * Cast Conductor Proprietary License v5 ("License"). By installing, copying,
 * accessing, compiling, or otherwise using the Software you agree to be bound by
 * all terms of the License. If you do not agree, you must cease use immediately.
 * 
 * Key Terms (Summary – see full License for binding terms):
 *  1. No Redistribution: You may not publish, distribute, sublicense, rent,
 *     lease, transfer, sell, or otherwise make the Software (or any derivative)
 *     available to any third party without prior written consent of Company.
 *  2. No Modification: Modification, reverse engineering, decompilation, or
 *     disassembly is prohibited except to the limited extent expressly permitted
 *     by applicable law that cannot be contractually waived.
 *  3. Confidentiality: Treat all source code and related artifacts as Company
 *     Confidential Information. Maintain at least the same degree of care as for
 *     your own confidential materials, and not less than reasonable care.
 *  4. No Patent License: No express or implied patent rights are granted. Future
 *     patents (if any) are fully reserved.
 *  5. No Trademark License: Company names, marks, and logos may not be used
 *     without prior written permission.
 *  6. Limited Internal Use: Use is limited solely to internal evaluation and
 *     operation of licensed Cast Conductor deployments. Commercial hosting or
 *     resale as a service requires a separate written agreement.
 *  7. Telemetry & License Validation: The Software may periodically transmit a
 *     hashed installation identifier, domain (or site ID), plugin/app version,
 *     and a truncated (non-reversible) fragment of the license key solely to
 *     validate activation status and enforce licensing. This minimal "phone home"
 *     check contains no personal or content data. If optional telemetry is later
 *     introduced it will be limited to aggregate operational metrics (no PII),
 *     fully documented, and optionally disableable per published instructions.
 *  8. Third-Party Components: The Software may include open source components
 *     covered by their own licenses. See THIRD-PARTY-NOTICES.md. Those licenses
 *     govern their respective components; this License governs all remaining code.
 *  9. Export Compliance: You are responsible for compliance with all applicable
 *     export control and sanctions laws.
 * 10. Warranty Disclaimer: THE SOFTWARE IS PROVIDED "AS IS" WITHOUT WARRANTY OF
 *     ANY KIND, EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO MERCHANTABILITY,
 *     FITNESS FOR A PARTICULAR PURPOSE, TITLE, AND NON-INFRINGEMENT.
 * 11. Limitation of Liability: IN NO EVENT WILL COMPANY OR AUTHORS BE LIABLE FOR
 *     ANY INDIRECT, SPECIAL, INCIDENTAL, CONSEQUENTIAL, EXEMPLARY, OR PUNITIVE
 *     DAMAGES, OR LOST PROFITS, EVEN IF ADVISED OF THE POSSIBILITY.
 * 12. Acceptance: Use of the Software constitutes acceptance of the License.
 * 13. Enforcement: Unauthorized reproduction or distribution may result in civil
 *     and criminal penalties and will be prosecuted to the maximum extent allowed
 *     by law.
 * 
 * Authoritative EULA: EULA-v5.1.md (repository root – private) and https://castconductor.com/eula
 * Precedence: If this summary conflicts with the EULA, the EULA governs.
 * Revision: Current EULA revision v5.1 (subject to update; check EULA for current enterprise thresholds).
 * 
 * Full License text available from: licensing@castconductor.com
 * Security reports: security@castconductor.com
 * Commercial inquiries: licensing@castconductor.com
 * 
 * END OF HEADER
 * 
 * Notification system extraction (Phase 2.7 Step 6)
 */

const DEFAULT_TTL = 5000; // ms
const MAX_VISIBLE = 4;

/**
 * Show a transient toast notification.
 * @param {Object} editor - editor instance (used for focusInvalid callback).
 * @param {string} message - text to display.
 * @param {string} type - info|success|warning|error
 * @param {Object} opts - { focusInvalid:boolean, ttl:number, dedupe:boolean }
 */
export function showNotification(editor, message, type = 'info', opts = {}) {
	if (!message) return;
	const { focusInvalid = false, ttl = DEFAULT_TTL, dedupe = true } = opts;
	const container = document.getElementById('canvas-notifications') || document.body;

	// Dedupe identical visible messages (optional)
	if (dedupe) {
		const existing = Array.from(container.querySelectorAll('.canvas-notification'))
			.find(el => el.textContent === message && el.dataset.type === type);
		if (existing) {
			// Refresh its timer by moving to end and updating removal timeout
			existing._ccveCancel && existing._ccveCancel();
			container.appendChild(existing);
			existing._ccveCancel = scheduleRemoval(existing, ttl);
			return;
		}
	}

	const el = document.createElement('div');
	el.className = `canvas-notification canvas-notification-${type}`;
	el.setAttribute('role', type === 'error' ? 'alert' : 'status');
	el.dataset.type = type;
	el.textContent = message;

	container.appendChild(el);

	// Enforce max visible (FIFO removal)
	const visibles = container.querySelectorAll('.canvas-notification');
	if (visibles.length > MAX_VISIBLE) {
		const first = visibles[0];
		if (first && first !== el) first.remove();
	}

	if (focusInvalid && editor && typeof editor.focusFirstInvalidInput === 'function') {
		try { editor.focusFirstInvalidInput(); } catch(_) {}
	}

	el._ccveCancel = scheduleRemoval(el, ttl);
}

function scheduleRemoval(el, ttl){
	const to = setTimeout(()=>{ try { el.remove(); } catch(_){} }, ttl);
	return () => clearTimeout(to);
}

export function hideNotifications() {
	// Clear known container
	const container = document.getElementById('canvas-notifications');
	if (container) container.innerHTML='';
	// Remove stray toasts
	document.querySelectorAll('.canvas-notification').forEach(el=>el.remove());
}

export const notificationsApi = { showNotification, hideNotifications };
