/**
 * Cast Conductor Proprietary License v5
 * SPDX-License-Identifier: LicenseRef-CastConductor-Proprietary-v5
 * 
 * Copyright (c) 2025 CastConductor.com. All Rights Reserved.
 * 
 * This file is part of Cast Conductor ("Software"). The Software and its source
 * code constitute proprietary, confidential, and trade secret information of
 * CastConductor.com ("Company"). Any access or use is governed strictly by the
 * Cast Conductor Proprietary License v5 ("License"). By installing, copying,
 * accessing, compiling, or otherwise using the Software you agree to be bound by
 * all terms of the License. If you do not agree, you must cease use immediately.
 * 
 * Key Terms (Summary – see full License for binding terms):
 *  1. No Redistribution: You may not publish, distribute, sublicense, rent,
 *     lease, transfer, sell, or otherwise make the Software (or any derivative)
 *     available to any third party without prior written consent of Company.
 *  2. No Modification: Modification, reverse engineering, decompilation, or
 *     disassembly is prohibited except to the limited extent expressly permitted
 *     by applicable law that cannot be contractually waived.
 *  3. Confidentiality: Treat all source code and related artifacts as Company
 *     Confidential Information. Maintain at least the same degree of care as for
 *     your own confidential materials, and not less than reasonable care.
 *  4. No Patent License: No express or implied patent rights are granted. Future
 *     patents (if any) are fully reserved.
 *  5. No Trademark License: Company names, marks, and logos may not be used
 *     without prior written permission.
 *  6. Limited Internal Use: Use is limited solely to internal evaluation and
 *     operation of licensed Cast Conductor deployments. Commercial hosting or
 *     resale as a service requires a separate written agreement.
 *  7. Telemetry & License Validation: The Software may periodically transmit a
 *     hashed installation identifier, domain (or site ID), plugin/app version,
 *     and a truncated (non-reversible) fragment of the license key solely to
 *     validate activation status and enforce licensing. This minimal "phone home"
 *     check contains no personal or content data. If optional telemetry is later
 *     introduced it will be limited to aggregate operational metrics (no PII),
 *     fully documented, and optionally disableable per published instructions.
 *  8. Third-Party Components: The Software may include open source components
 *     covered by their own licenses. See THIRD-PARTY-NOTICES.md. Those licenses
 *     govern their respective components; this License governs all remaining code.
 *  9. Export Compliance: You are responsible for compliance with all applicable
 *     export control and sanctions laws.
 * 10. Warranty Disclaimer: THE SOFTWARE IS PROVIDED "AS IS" WITHOUT WARRANTY OF
 *     ANY KIND, EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO MERCHANTABILITY,
 *     FITNESS FOR A PARTICULAR PURPOSE, TITLE, AND NON-INFRINGEMENT.
 * 11. Limitation of Liability: IN NO EVENT WILL COMPANY OR AUTHORS BE LIABLE FOR
 *     ANY INDIRECT, SPECIAL, INCIDENTAL, CONSEQUENTIAL, EXEMPLARY, OR PUNITIVE
 *     DAMAGES, OR LOST PROFITS, EVEN IF ADVISED OF THE POSSIBILITY.
 * 12. Acceptance: Use of the Software constitutes acceptance of the License.
 * 13. Enforcement: Unauthorized reproduction or distribution may result in civil
 *     and criminal penalties and will be prosecuted to the maximum extent allowed
 *     by law.
 * 
 * Authoritative EULA: EULA-v5.1.md (repository root – private) and https://castconductor.com/eula
 * Precedence: If this summary conflicts with the EULA, the EULA governs.
 * Revision: Current EULA revision v5.1 (subject to update; check EULA for current enterprise thresholds).
 * 
 * Full License text available from: licensing@castconductor.com
 * Security reports: security@castconductor.com
 * Commercial inquiries: licensing@castconductor.com
 * 
 * END OF HEADER
 */
export const LOGICAL_STAGE_WIDTH = 1920;
export const LOGICAL_STAGE_HEIGHT = 1080;

/** Compute a uniform scale that fits the logical stage into the available box */
export function computeUniformScale(availableWidth, availableHeight, logicalW = LOGICAL_STAGE_WIDTH, logicalH = LOGICAL_STAGE_HEIGHT) {
	const sx = availableWidth / logicalW;
	const sy = availableHeight / logicalH;
	return Math.min(sx, sy);
}

/**
 * Apply scale transform to a stage element (top-left origin) and optionally return applied scale.
 * Expects the stage element already has explicit logical width/height set in pixels.
 */
export function applySurfaceScale(stageEl, availableWidth, availableHeight, logicalW = LOGICAL_STAGE_WIDTH, logicalH = LOGICAL_STAGE_HEIGHT) {
	if (!stageEl) return 1;
	const scale = computeUniformScale(availableWidth, availableHeight, logicalW, logicalH);
	stageEl.style.transformOrigin = 'top left';
	stageEl.style.transform = `scale(${scale})`;
	return scale;
}

/** Clamp a value between min & max */
export function clamp(v, min, max){ return Math.max(min, Math.min(max, v)); }

/** Snap a value to the nearest step (defaults to 8) */
export function snap(v, step = 8){ return Math.round(v / step) * step; }

/** Convenience rectangle object builder */
export function rect(x=0,y=0,w=0,h=0){ return {x,y,w,h}; }

/** Returns true if a rectangle is inside bounds (optionally allowing partial overlap) */
export function rectWithin(r, bounds, allowPartial=false){
	if (!r || !bounds) return false;
	if (allowPartial) {
		return !(r.x + r.w < 0 || r.y + r.h < 0 || r.x > bounds.w || r.y > bounds.h);
	}
	return r.x >= 0 && r.y >= 0 && (r.x + r.w) <= bounds.w && (r.y + r.h) <= bounds.h;
}

/** Normalize geometry object (coerces to integers & clamps dimensions) */
export function normalizeRect(r, bounds){
	if (!r) return rect();
	const out = rect(parseInt(r.x,10)||0, parseInt(r.y,10)||0, parseInt(r.w,10)||parseInt(r.width,10)||0, parseInt(r.h,10)||parseInt(r.height,10)||0);
	if (bounds) {
		out.w = clamp(out.w, 1, bounds.w);
		out.h = clamp(out.h, 1, bounds.h);
		out.x = clamp(out.x, 0, bounds.w - out.w);
		out.y = clamp(out.y, 0, bounds.h - out.h);
	}
	return out;
}

/** Derive available width/height from a stage's parent element */
export function getAvailableBox(stageEl){
	if (!stageEl || !stageEl.parentElement) return { w: LOGICAL_STAGE_WIDTH, h: LOGICAL_STAGE_HEIGHT };
	const parent = stageEl.parentElement;
	const rect = parent.getBoundingClientRect();
	return { w: rect.width, h: rect.height };
}

// Global namespace bridge (temporary) so legacy monolith can call without ES module import bundling.
// Remove once build tooling / module loader is in place.
if (typeof window !== 'undefined') {
	window.ccveSharedGeometry = {
		LOGICAL_STAGE_WIDTH,
		LOGICAL_STAGE_HEIGHT,
		computeUniformScale,
		applySurfaceScale,
		clamp,
		snap,
		rect,
		rectWithin,
		normalizeRect,
		getAvailableBox
	};
}

