/**
 * Token Insertion Modal (Phase 4 – initial minimal implementation)
 * Provides lightweight UI to build a template containing {{category.field}} tokens
 * and insert it as a new text layer inside the current content block visual config.
 *
 * Scope (Phase 4 initial):
 *  - Lists categories & fields from token registry.
 *  - Clicking a field appends token syntax to a working <textarea>.
 *  - User can manually edit order / add separators.
 *  - "Insert Layer" creates a basic positioned text layer (stacking offset) with templateText.
 *  - Immediately triggers refreshLiveTokens() to render live values.
 *  - No reposition / resize UI yet (future enhancement) – layers placed with incremental y offsets.
 */

import { listCategories, listFields, getTokenMeta, isValidToken } from './token-registry.js';

export function openTokenInsertModal(editor) {
  if (!editor || typeof editor.showModal !== 'function') return;
  const cats = listCategories();
  const buildCategoryList = () => {
    return `<div class="tim-cats">${cats.map(c=>`<button type="button" class="tim-cat-btn" data-cat="${c}">${c}</button>`).join('')}</div>`;
  };
  const buildFields = (cat) => {
    if (!cat) return '<div class="tim-fields">Select a category.</div>';
    const fields = listFields(cat);
    return `<div class="tim-fields" data-fields-for="${cat}">` + fields.map(f=>{
      const meta = getTokenMeta(`${cat}.${f}`) || {}; const desc = meta.description || '';
      return `<button type="button" class="tim-field-btn" data-token="${cat}.${f}" title="${desc}">{{${cat}.${f}}}</button>`;
    }).join('') + '</div>';
  };
  const initialCat = cats[0];
  const html = `
  <style>
    .tim-wrapper { display:flex; flex-direction:column; gap:8px; font:12px system-ui,Arial; }
    .tim-cats { display:flex; flex-wrap:wrap; gap:4px; }
    .tim-cats .tim-cat-btn { background:#1e293b; color:#fff; border:1px solid #334155; padding:4px 8px; border-radius:4px; cursor:pointer; font-size:11px; }
    .tim-fields { display:flex; flex-wrap:wrap; gap:4px; max-height:120px; overflow:auto; }
    .tim-field-btn { background:#0f172a; color:#fff; border:1px solid #334155; padding:4px 6px; border-radius:4px; cursor:pointer; font-size:11px; }
    .tim-template-box { width:100%; min-height:70px; font:12px/1.4 monospace; padding:6px; resize:vertical; }
    .tim-actions { display:flex; gap:8px; justify-content:space-between; align-items:center; margin-top:4px; }
    .tim-insert-btn { background:#2563eb; color:#fff; border:1px solid #1d4ed8; padding:6px 14px; border-radius:4px; cursor:pointer; font-weight:600; font-size:12px; }
    .tim-cancel-btn { background:#64748b; color:#fff; border:1px solid #475569; padding:6px 10px; border-radius:4px; cursor:pointer; font-size:12px; }
    .tim-hint { font-size:11px; color:#94a3b8; }
  </style>
  <div class="tim-wrapper" id="ccve-token-insert-modal">
    <div><strong>1. Choose tokens</strong></div>
    ${buildCategoryList()}
    <div id="tim-fields-container">${buildFields(initialCat)}</div>
    <div><strong>2. Build template</strong> <span class="tim-hint">Click tokens to append; you may type text & separators (e.g. •, —)</span></div>
    <textarea id="tim-template" class="tim-template-box" placeholder="Example: {{track.title}} — {{track.artist}}"></textarea>
    <div class="tim-actions">
      <div class="tim-hint">Result becomes a new text layer with live tokens.</div>
      <div>
        <button type="button" class="tim-cancel-btn" id="tim-cancel">Cancel</button>
        <button type="button" class="tim-insert-btn" id="tim-insert">Insert Layer</button>
      </div>
    </div>
  </div>`;
  editor.showModal('Insert Live Tokens', html);
  const modalRoot = document.getElementById('ccve-token-insert-modal');
  if (!modalRoot) return;
  const tplBox = modalRoot.querySelector('#tim-template');
  const fieldsContainer = modalRoot.querySelector('#tim-fields-container');
  // Category switching
  modalRoot.querySelectorAll('.tim-cat-btn').forEach(btn=>btn.addEventListener('click',()=>{
    const cat = btn.getAttribute('data-cat');
    fieldsContainer.innerHTML = buildFields(cat);
    fieldsContainer.querySelectorAll('.tim-field-btn').forEach(fb=>fb.addEventListener('click',()=>{
      const token = fb.getAttribute('data-token');
      if (token) {
        const insertion = `{{${token}}}`;
        tplBox.value = tplBox.value ? (tplBox.value.trimEnd() + ' ' + insertion) : insertion;
        tplBox.focus();
      }
    }));
  }));
  // Bind initial field buttons
  fieldsContainer.querySelectorAll('.tim-field-btn').forEach(fb=>fb.addEventListener('click',()=>{
    const token = fb.getAttribute('data-token');
    if (token) {
      const insertion = `{{${token}}}`;
      tplBox.value = tplBox.value ? (tplBox.value.trimEnd() + ' ' + insertion) : insertion;
      tplBox.focus();
    }
  }));
  // Cancel
  modalRoot.querySelector('#tim-cancel').addEventListener('click', ()=> editor.closeModal());
  // Insert
  modalRoot.querySelector('#tim-insert').addEventListener('click', ()=>{
    const raw = (tplBox.value || '').trim();
    if (!raw) { tplBox.focus(); return; }
    // Validate at least one token present
    const tokenMatches = Array.from(raw.matchAll(/\{\{\s*([a-z0-9_]+\.[a-z0-9_]+)\s*\}\}/ig)).map(m=>m[1]);
    const valid = tokenMatches.filter(t=>isValidToken(t));
    if (!valid.length) { tplBox.focus(); return; }
    // Guardrail: route image-type tokens (e.g., artwork) to image layers instead of text URLs
    const imageTokens = valid.filter(t=>{ const meta = getTokenMeta(t); return meta && meta.type === 'image'; });
    const textTokens = valid.filter(t=>{ const meta = getTokenMeta(t); return meta && meta.type !== 'image'; });
    // Helper to strip selected tokens from the raw template
    const stripTokensFromTemplate = (template, tokens)=>{
      let out = template;
      for (const t of tokens) {
        const re = new RegExp(`\\{\\{\\s*${t.replace('.', '\\.')}\\s*\\}\\}`, 'ig');
        out = out.replace(re, '').replace(/\\s{2,}/g,' ').trim();
      }
      return out;
    };
    // If any image tokens exist, insert image layers for them and remove from text template
    if (imageTokens.length) {
      try { for (const it of imageTokens) { editor.addTokenImageLayer(it); } } catch(e) { console.warn('addTokenImageLayer failed', e); }
    }
    const cleaned = imageTokens.length ? stripTokensFromTemplate(raw, imageTokens) : raw;
    const hasMeaningfulText = cleaned && cleaned.replace(/[\s•—\-_,.:;|/\\]+/g,'').length > 0;
    if (hasMeaningfulText && (textTokens.length || !imageTokens.length)) {
      try { editor.addTokenTextLayer(cleaned); } catch(e) { console.warn('addTokenTextLayer failed', e); }
    }
    if (imageTokens.length) {
      try { editor.showNotification && editor.showNotification('Artwork tokens were added as image layers; text layer updated accordingly.', 'info'); } catch(_){ }
    }
    editor.closeModal();
    try { editor.refreshLiveTokens(); } catch(_){}
  });
}

export default { openTokenInsertModal };
