/**
 * Token Registry (V5 Editor Live Token Rebuild – Phase 1)
 * Maps high-level token paths (category.field) to normalized internal field keys.
 * No network logic here – purely declarative so other modules can look up
 * allowed tokens and derive categories/fields for UI listing.
 *
 * Weather / Location tokens: We surface styling placeholders in the admin
 * editor only; Roku runtime will render true GEO/IP derived values.
 * 
 * Custom API tokens: Dynamically mapped fields from external APIs (Phase 1 MVP).
 */

const _registry = {
  track: {
    title: { type: 'text', sourceField: 'title', description: 'Current track title' },
    artist: { type: 'text', sourceField: 'artist', description: 'Current track artist' },
    album: { type: 'text', sourceField: 'album', description: 'Current track album' },
    artwork: { type: 'image', sourceField: 'artwork_url', description: 'Album / track artwork URL' }
  },
  weather: {
    temperature: { type: 'text', sourceField: 'temperature', description: 'Viewer temperature (IP localized at runtime)' },
    condition: { type: 'text', sourceField: 'condition', description: 'Weather condition (IP localized at runtime)' },
    unit: { type: 'text', sourceField: 'unit', description: 'Temperature unit (°F or °C)' }
  },
  location: {
    time: { type: 'text', sourceField: 'time', description: 'Local viewer time (runtime device localized)' },
    date: { type: 'text', sourceField: 'date', description: 'Local viewer date (runtime device localized)' },
    city: { type: 'text', sourceField: 'city', description: 'Viewer city (runtime device localized)' },
    state: { type: 'text', sourceField: 'state', description: 'Viewer state/region (runtime device localized)' }
  },
  shoutout: {
    name: { type: 'text', sourceField: 'name', description: 'Shoutout name' },
    location: { type: 'text', sourceField: 'location', description: 'Shoutout location' },
    message: { type: 'text', sourceField: 'message', description: 'Shoutout message body' },
    artwork: { type: 'image', sourceField: 'artwork_url', description: 'Shoutout image (selfie, GIF, or emoji)' }
  },
  sponsor: {
    title: { type: 'text', sourceField: 'title', description: 'Sponsor title' },
    content: { type: 'text', sourceField: 'content', description: 'Sponsor body/content' },
    artwork: { type: 'image', sourceField: 'featured_media_url', description: 'Sponsor image / logo' }
  },
  promo: {
    title: { type: 'text', sourceField: 'title', description: 'Promo title' },
    content: { type: 'text', sourceField: 'content', description: 'Promo body/content' },
    artwork: { type: 'image', sourceField: 'featured_media_url', description: 'Promo image' }
  },
  custom: {
    text: { type: 'text', sourceField: 'content', description: 'Manual custom text' }
  },
  // Custom API tokens (Phase 1 MVP - Dec 2025)
  // These map to JSONPath-extracted fields from external APIs
  custom_api: {
    primary: { type: 'text', sourceField: 'primary_text', description: 'Primary text from API (mapped via JSONPath)' },
    secondary: { type: 'text', sourceField: 'secondary_text', description: 'Secondary text from API (mapped via JSONPath)' },
    number: { type: 'text', sourceField: 'number_display', description: 'Formatted number/value from API (with prefix/suffix)' },
    date: { type: 'text', sourceField: 'date_value', description: 'Date/time value from API' },
    image: { type: 'image', sourceField: 'image_url', description: 'Image URL from API (mapped via JSONPath)' },
    link: { type: 'text', sourceField: 'link_url', description: 'Link URL from API (mapped via JSONPath)' }
  }
};

export function getRegistry() { return _registry; }
export function listCategories() { return Object.keys(_registry); }
export function listFields(category) { return _registry[category] ? Object.keys(_registry[category]) : []; }
export function isValidToken(path) {
  if (!path || typeof path !== 'string') return false;
  const parts = path.split('.');
  if (parts.length !== 2) return false;
  const [cat, field] = parts;
  return !!(_registry[cat] && _registry[cat][field]);
}
export function getTokenMeta(path) {
  if (!isValidToken(path)) return null;
  const [cat, field] = path.split('.');
  return { category: cat, field, ..._registry[cat][field] };
}

// Simple helper to enumerate all tokens flat (for export or scanning)
export function listAllTokens() {
  const out = [];
  for (const c of listCategories()) {
    for (const f of listFields(c)) out.push(`${c}.${f}`);
  }
  return out;
}

// Attach to window for quick manual inspection during early development.
try { window.ccveTokenRegistry = { getRegistry, listCategories, listFields, isValidToken, getTokenMeta, listAllTokens }; } catch(_) {}

export default _registry;
