<?php
/**
 * Plugin Name: Cast Conductor V5
 * Plugin URI: https://castconductor.com
 * Description: Professional Roku content management system for broadcasters, content creators, and businesses. Provides real-time content blocks, scheduling, and dynamic media displays.
 * Version: 5.7.25-202601141142
 * Author: CastConductor.com
 * License: Proprietary (See EULA-v5.3.md) + OSS components (THIRD-PARTY-NOTICES.md)
 * Text Domain: castconductor
 * Domain Path: /languages
 * Requires at least: 5.0
 * Tested up to: 6.9
 * Requires PHP: 7.4
 * Network: false
 *
 * Cast Conductor Proprietary License v5
 * SPDX-License-Identifier: LicenseRef-CastConductor-Proprietary-v5
 *
 * Copyright (c) 2025 CastConductor.com. All Rights Reserved.
 *
 * This file is part of Cast Conductor ("Software"). The Software and its source code
 * constitute proprietary, confidential, and trade secret information of CastConductor.com
 * ("Company"). Any access or use is governed strictly by the Cast Conductor Proprietary
 * License v5 ("License"). By installing, copying, accessing, compiling, or otherwise
 * using the Software you agree to be bound by all terms of the License. If you do not
 * agree, you must cease use immediately.
 *
 * Key Terms (Summary – see full License for binding terms):
 *  1. No Redistribution: You may not publish, distribute, sublicense, rent, lease, transfer,
 *     sell, or otherwise make the Software (or any derivative) available to any third party
 *     without prior written consent of Company.
 *  2. No Modification: Modification, reverse engineering, decompilation, or disassembly is
 *     prohibited except to the limited extent expressly permitted by applicable law that
 *     cannot be contractually waived.
 *  3. Confidentiality: Treat all source code and related artifacts as Company Confidential
 *     Information. Maintain at least the same degree of care as for your own confidential
 *     materials, and not less than reasonable care.
 *  4. No Patent License: No express or implied patent rights are granted. Future patents
 *     (if any) are fully reserved.
 *  5. No Trademark License: Company names, marks, and logos may not be used without prior
 *     written permission.
 *  6. Limited Internal Use: Use is limited solely to internal evaluation and operation of
 *     licensed Cast Conductor deployments. Commercial hosting or resale as a service
 *     requires a separate written agreement.
 *  7. Telemetry & License Validation: The Software may periodically transmit a hashed
 *     installation identifier, domain (or site ID), plugin/app version, and a truncated
 *     (non-reversible) fragment of the license key solely to validate activation status and
 *     enforce licensing. This minimal "phone home" check contains no personal or content
 *     data. Optional telemetry (if introduced) will be aggregate only (no PII) and
 *     disableable per documentation.
 *  8. Third-Party Components: The Software may include open source components covered by
 *     their own licenses. See THIRD-PARTY-NOTICES.md. Those licenses govern their
 *     respective components; this License governs all remaining code.
 *  9. Export Compliance: You are responsible for compliance with all applicable export
 *     control and sanctions laws.
 * 10. Warranty Disclaimer: THE SOFTWARE IS PROVIDED "AS IS" WITHOUT WARRANTY OF ANY KIND,
 *     EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO MERCHANTABILITY, FITNESS FOR A
 *     PARTICULAR PURPOSE, TITLE, AND NON-INFRINGEMENT.
 * 11. Limitation of Liability: IN NO EVENT WILL COMPANY OR AUTHORS BE LIABLE FOR ANY
 *     INDIRECT, SPECIAL, INCIDENTAL, CONSEQUENTIAL, EXEMPLARY, OR PUNITIVE DAMAGES, OR
 *     LOST PROFITS, EVEN IF ADVISED OF THE POSSIBILITY.
 * 12. Acceptance: Use of the Software constitutes acceptance of the License.
 * 13. Enforcement: Unauthorized reproduction or distribution may result in civil and
 *     criminal penalties and will be prosecuted to the maximum extent allowed by law.
 *
 * Authoritative EULA: EULA-v5.1.md (repository root – private) and https://castconductor.com/eula
 * Precedence: If this summary conflicts with the EULA, the EULA governs.
 * Revision: Current EULA revision v5.1 (subject to update; check EULA for current enterprise thresholds).
 *
 * Full License text available from: licensing@castconductor.com
 * Security reports: security@castconductor.com
 * Commercial inquiries: licensing@castconductor.com
 *
 * END OF HEADER
 */
// Prevent direct access
if (!defined('ABSPATH')) {
    exit;
}

// Define plugin constants
define('CASTCONDUCTOR_VERSION', '5.7.25-202601141142');
define('CASTCONDUCTOR_BUILD', 279);
define('CASTCONDUCTOR_PLUGIN_FILE', __FILE__);
define('CASTCONDUCTOR_PLUGIN_DIR', plugin_dir_path(__FILE__));
define('CASTCONDUCTOR_PLUGIN_URL', plugin_dir_url(__FILE__));
define('CASTCONDUCTOR_PLUGIN_BASENAME', plugin_basename(__FILE__));

// Minimum requirements
define('CASTCONDUCTOR_MIN_PHP_VERSION', '7.4');
define('CASTCONDUCTOR_MIN_WP_VERSION', '5.0');

/**
 * Main CastConductor Plugin Class
 *
 * Implements clean architecture with WordPress-compliant naming conventions
 * Creates production-ready content structure on activation
 */
class CastConductor_Plugin {
    /**
     * Single instance of the plugin
     */
    private static $instance = null;

    /**
     * Plugin components
     */
    public $database;
    public $rest_api;
    public $admin;
    public $content_blocks;
    public $containers;
    public $activation_wizard;
    public $license_manager;
    public $update_checker;
    public $advertising;
    public $analytics;

    /**
     * Get single instance
     */
    public static function instance() {
        if (null === self::$instance) {
            self::$instance = new self();
        }
        return self::$instance;
    }

    /**
     * Constructor - Initialize plugin
     */
    private function __construct() {
        add_action('plugins_loaded', array($this, 'init'));
        register_activation_hook(__FILE__, array($this, 'activate'));
        register_deactivation_hook(__FILE__, array($this, 'deactivate'));
        register_uninstall_hook(__FILE__, array('CastConductor_Plugin', 'uninstall'));
    }

    /**
     * Initialize plugin after WordPress loads
     */
    public function init() {
        // Check requirements
        if (!$this->check_requirements()) {
            return;
        }
        // Load dependencies
        $this->load_dependencies();
        
        // Initialize components
        $this->init_components();
        
        // Setup hooks
        $this->setup_hooks();
        
        // Load text domain
            load_plugin_textdomain('castconductor', false, dirname(CASTCONDUCTOR_PLUGIN_BASENAME) . '/languages');
    }
    
    /**
     * Check minimum requirements
     */
    private function check_requirements() {
        // Check PHP version
        if (version_compare(PHP_VERSION, CASTCONDUCTOR_MIN_PHP_VERSION, '<')) {
            add_action('admin_notices', array($this, 'php_version_notice'));
            return false;
        }
        
        // Check WordPress version
        global $wp_version;
        if (version_compare($wp_version, CASTCONDUCTOR_MIN_WP_VERSION, '<')) {
            add_action('admin_notices', array($this, 'wp_version_notice'));
            return false;
        }
        
        return true;
    }
    
    /**
     * Load required files
     */
    private function load_dependencies() {
        // Core classes
        require_once CASTCONDUCTOR_PLUGIN_DIR . 'includes/class-castconductor-database.php';
        require_once CASTCONDUCTOR_PLUGIN_DIR . 'includes/class-castconductor-rest-api.php';
        require_once CASTCONDUCTOR_PLUGIN_DIR . 'includes/class-castconductor-admin.php';
        require_once CASTCONDUCTOR_PLUGIN_DIR . 'includes/class-castconductor-content-blocks.php';
        require_once CASTCONDUCTOR_PLUGIN_DIR . 'includes/class-castconductor-containers.php';
        require_once CASTCONDUCTOR_PLUGIN_DIR . 'includes/class-castconductor-activation-wizard.php';
        require_once CASTCONDUCTOR_PLUGIN_DIR . 'includes/class-castconductor-shortcodes.php';
        
        // Content block types
        require_once CASTCONDUCTOR_PLUGIN_DIR . 'includes/content-blocks/class-track-info-block.php';
        require_once CASTCONDUCTOR_PLUGIN_DIR . 'includes/content-blocks/class-weather-block.php';
        require_once CASTCONDUCTOR_PLUGIN_DIR . 'includes/content-blocks/class-location-time-block.php';
        require_once CASTCONDUCTOR_PLUGIN_DIR . 'includes/content-blocks/class-shoutout-block.php';
        require_once CASTCONDUCTOR_PLUGIN_DIR . 'includes/content-blocks/class-sponsor-block.php';
        require_once CASTCONDUCTOR_PLUGIN_DIR . 'includes/content-blocks/class-promo-block.php';
        require_once CASTCONDUCTOR_PLUGIN_DIR . 'includes/content-blocks/class-custom-api-block.php';
        require_once CASTCONDUCTOR_PLUGIN_DIR . 'includes/content-blocks/class-custom-block.php';
        
        // API controllers
        require_once CASTCONDUCTOR_PLUGIN_DIR . 'includes/api/class-roku-controller.php';
        require_once CASTCONDUCTOR_PLUGIN_DIR . 'includes/api/class-containers-controller.php';
        require_once CASTCONDUCTOR_PLUGIN_DIR . 'includes/api/class-background-controller.php';
        require_once CASTCONDUCTOR_PLUGIN_DIR . 'includes/api/class-content-blocks-controller.php';
        require_once CASTCONDUCTOR_PLUGIN_DIR . 'includes/api/class-assignments-controller.php';
        require_once CASTCONDUCTOR_PLUGIN_DIR . 'includes/api/class-content-controller.php';
        require_once CASTCONDUCTOR_PLUGIN_DIR . 'includes/api/class-artwork-controller.php';
        require_once CASTCONDUCTOR_PLUGIN_DIR . 'includes/api/class-album-artwork-controller.php';
        require_once CASTCONDUCTOR_PLUGIN_DIR . 'includes/api/class-metadata-proxy-controller.php';
        require_once CASTCONDUCTOR_PLUGIN_DIR . 'includes/api/class-fetch-proxy-controller.php';
        require_once CASTCONDUCTOR_PLUGIN_DIR . 'includes/api/class-canvas-editor-controller.php';
        require_once CASTCONDUCTOR_PLUGIN_DIR . 'includes/api/class-shoutout-moderation-controller.php';
        require_once CASTCONDUCTOR_PLUGIN_DIR . 'includes/api/class-configuration-backup-controller.php';
        require_once CASTCONDUCTOR_PLUGIN_DIR . 'includes/api/class-live-data-integration-controller.php';
        require_once CASTCONDUCTOR_PLUGIN_DIR . 'includes/api/class-settings-controller.php';
        require_once CASTCONDUCTOR_PLUGIN_DIR . 'includes/api/class-scenes-controller.php';
        require_once CASTCONDUCTOR_PLUGIN_DIR . 'includes/api/class-feed-controller.php';
        require_once CASTCONDUCTOR_PLUGIN_DIR . 'includes/api/class-deep-link-controller.php';
        require_once CASTCONDUCTOR_PLUGIN_DIR . 'includes/api/class-menu-controller.php';
        
        // CRUD Controllers for V5 Unified Layer System
        // Note: Duplicate class-castconductor-*-controller.php files removed - using includes/api/ versions instead
        // Removed: class-castconductor-content-blocks-controller.php (duplicate of includes/api/class-content-blocks-controller.php)
        // Removed: class-castconductor-scenes-controller.php (duplicate of includes/api/class-scenes-controller.php)
        // Removed: class-castconductor-containers-controller.php (duplicate of includes/api/class-containers-controller.php)
        
        // Admin pages
        require_once CASTCONDUCTOR_PLUGIN_DIR . 'includes/admin/class-shoutout-moderation-admin.php';
        require_once CASTCONDUCTOR_PLUGIN_DIR . 'includes/admin/class-castconductor-export-import-page.php';
        require_once CASTCONDUCTOR_PLUGIN_DIR . 'includes/admin/class-menu-builder-admin.php';
        
        // Export/Import managers
        require_once CASTCONDUCTOR_PLUGIN_DIR . 'includes/class-castconductor-export-manager.php';
        require_once CASTCONDUCTOR_PLUGIN_DIR . 'includes/class-castconductor-import-manager.php';
        
        // Utilities
        require_once CASTCONDUCTOR_PLUGIN_DIR . 'includes/utils/class-artwork-search.php';
        require_once CASTCONDUCTOR_PLUGIN_DIR . 'includes/utils/class-geolocation.php';
        require_once CASTCONDUCTOR_PLUGIN_DIR . 'includes/utils/class-url-config.php';
        require_once CASTCONDUCTOR_PLUGIN_DIR . 'includes/utils/class-feed-parser.php';
        
        // License Management
        require_once CASTCONDUCTOR_PLUGIN_DIR . 'includes/class-castconductor-license-manager.php';
        
        // Update Checker
        require_once CASTCONDUCTOR_PLUGIN_DIR . 'includes/class-castconductor-update-checker.php';
        
        // Advertising (Roku Ads Integration)
        require_once CASTCONDUCTOR_PLUGIN_DIR . 'includes/class-castconductor-advertising.php';
        
        // Debug Logging System
        require_once CASTCONDUCTOR_PLUGIN_DIR . 'includes/class-castconductor-debug.php';
        
        // Analytics Module (v5.8.0+)
        require_once CASTCONDUCTOR_PLUGIN_DIR . 'includes/analytics/class-cc-analytics-helpers.php';
        require_once CASTCONDUCTOR_PLUGIN_DIR . 'includes/analytics/class-cc-analytics-database.php';
        require_once CASTCONDUCTOR_PLUGIN_DIR . 'includes/analytics/class-cc-analytics-ingestion.php';
        require_once CASTCONDUCTOR_PLUGIN_DIR . 'includes/analytics/class-cc-analytics-aggregation.php';
        require_once CASTCONDUCTOR_PLUGIN_DIR . 'includes/analytics/class-cc-analytics-api.php';
        require_once CASTCONDUCTOR_PLUGIN_DIR . 'includes/analytics/class-cc-analytics-dashboard.php';
        require_once CASTCONDUCTOR_PLUGIN_DIR . 'includes/analytics/class-cc-analytics-integrations.php';
    }
    
    /**
     * Initialize plugin components
     */
    private function init_components() {
        $this->database = new CastConductor_Database();
        $this->rest_api = new CastConductor_REST_API();
        $this->admin = new CastConductor_Admin();
        $this->content_blocks = new CastConductor_Content_Blocks();
        $this->containers = new CastConductor_Containers();
        $this->activation_wizard = new CastConductor_Activation_Wizard();
        $this->license_manager = CastConductor_License_Manager::instance();
        $this->update_checker = CastConductor_Update_Checker::instance();
        $this->advertising = CastConductor_Advertising::instance();
        
        // Initialize Analytics Module
        $this->analytics = (object) [
            'database'     => CC_Analytics_Database::instance(),
            'ingestion'    => CC_Analytics_Ingestion::instance(),
            'aggregation'  => CC_Analytics_Aggregation::instance(),
            'api'          => CC_Analytics_API::instance(),
            'dashboard'    => CC_Analytics_Dashboard::instance(),
            'integrations' => CC_Analytics_Integrations::instance(),
        ];
    }
    
    /**
     * Setup WordPress hooks
     */
    private function setup_hooks() {
        add_action('init', array($this, 'register_post_types'));
        add_action('init', array($this, 'register_taxonomies'));
        add_action('rest_api_init', array($this->rest_api, 'register_routes'));
    }
    
    /**
     * Register custom post types with WordPress-compliant naming
     * Fixed: Post type names must be 20 characters or less
     */
    public function register_post_types() {
        // Get custom labels if set
        $custom_labels = get_option('castconductor_cpt_labels', array());
        
        // Helper function to get label with fallback
        $get_label = function($cpt, $type, $default) use ($custom_labels) {
            if (isset($custom_labels[$cpt][$type]) && !empty($custom_labels[$cpt][$type])) {
                return $custom_labels[$cpt][$type];
            }
            return $default;
        };
        
        // Shoutouts post type (cc_shoutout = 11 chars)
        $shoutout_name = $get_label('cc_shoutout', 'name', __('Shoutouts', 'castconductor'));
        $shoutout_singular = $get_label('cc_shoutout', 'singular', __('Shoutout', 'castconductor'));
        register_post_type('cc_shoutout', array(
            'labels' => array(
                'name' => $shoutout_name,
                'singular_name' => $shoutout_singular,
                /* translators: %s: singular name of the custom post type */
                'add_new' => sprintf(__('Add New %s', 'castconductor'), $shoutout_singular),
                /* translators: %s: singular name of the custom post type */
                'add_new_item' => sprintf(__('Add New %s', 'castconductor'), $shoutout_singular),
                /* translators: %s: singular name of the custom post type */
                'edit_item' => sprintf(__('Edit %s', 'castconductor'), $shoutout_singular),
                /* translators: %s: singular name of the custom post type */
                'new_item' => sprintf(__('New %s', 'castconductor'), $shoutout_singular),
                /* translators: %s: singular name of the custom post type */
                'view_item' => sprintf(__('View %s', 'castconductor'), $shoutout_singular),
                /* translators: %s: plural name of the custom post type */
                'search_items' => sprintf(__('Search %s', 'castconductor'), $shoutout_name),
                /* translators: %s: plural name - no items found in trash */
                'not_found' => sprintf(__('No %s found', 'castconductor'), strtolower($shoutout_name)),
                /* translators: %s: plural name - no items found in trash */
                'not_found_in_trash' => sprintf(__('No %s found in trash', 'castconductor'), strtolower($shoutout_name)),
            ),
            'public' => false,
            'show_ui' => true,
            'show_in_menu' => 'castconductor',
            'show_in_rest' => true,
            'supports' => array('title', 'editor', 'custom-fields', 'thumbnail'),
            'has_archive' => false,
            'rewrite' => false,
        ));
        
        // Sponsors post type (cc_sponsor = 10 chars)
        $sponsor_name = $get_label('cc_sponsor', 'name', __('Sponsors', 'castconductor'));
        $sponsor_singular = $get_label('cc_sponsor', 'singular', __('Sponsor', 'castconductor'));
        register_post_type('cc_sponsor', array(
            'labels' => array(
                'name' => $sponsor_name,
                'singular_name' => $sponsor_singular,
                'add_new' => sprintf(__('Add New %s', 'castconductor'), $sponsor_singular),
                'add_new_item' => sprintf(__('Add New %s', 'castconductor'), $sponsor_singular),
                'edit_item' => sprintf(__('Edit %s', 'castconductor'), $sponsor_singular),
                'new_item' => sprintf(__('New %s', 'castconductor'), $sponsor_singular),
                'view_item' => sprintf(__('View %s', 'castconductor'), $sponsor_singular),
                'search_items' => sprintf(__('Search %s', 'castconductor'), $sponsor_name),
                'not_found' => sprintf(__('No %s found', 'castconductor'), strtolower($sponsor_name)),
                'not_found_in_trash' => sprintf(__('No %s found in trash', 'castconductor'), strtolower($sponsor_name)),
            ),
            'public' => false,
            'show_ui' => true,
            'show_in_menu' => 'castconductor',
            'show_in_rest' => true,
            'supports' => array('title', 'editor', 'thumbnail', 'custom-fields'),
            'has_archive' => false,
            'rewrite' => false,
        ));
        
        // Promos post type (cc_promo = 8 chars)
        $promo_name = $get_label('cc_promo', 'name', __('Promos', 'castconductor'));
        $promo_singular = $get_label('cc_promo', 'singular', __('Promo', 'castconductor'));
        register_post_type('cc_promo', array(
            'labels' => array(
                'name' => $promo_name,
                'singular_name' => $promo_singular,
                'add_new' => sprintf(__('Add New %s', 'castconductor'), $promo_singular),
                'add_new_item' => sprintf(__('Add New %s', 'castconductor'), $promo_singular),
                'edit_item' => sprintf(__('Edit %s', 'castconductor'), $promo_singular),
                'new_item' => sprintf(__('New %s', 'castconductor'), $promo_singular),
                'view_item' => sprintf(__('View %s', 'castconductor'), $promo_singular),
                'search_items' => sprintf(__('Search %s', 'castconductor'), $promo_name),
                'not_found' => sprintf(__('No %s found', 'castconductor'), strtolower($promo_name)),
                'not_found_in_trash' => sprintf(__('No %s found in trash', 'castconductor'), strtolower($promo_name)),
            ),
            'public' => false,
            'show_ui' => true,
            'show_in_menu' => 'castconductor',
            'show_in_rest' => true,
            'supports' => array('title', 'editor', 'thumbnail', 'custom-fields'),
            'has_archive' => false,
            'rewrite' => false,
        ));
        
        // Layers post type for WordPress Post layers (cc_layer = 8 chars)
        // Used for creating custom content that displays as layers in the canvas
        $layer_name = $get_label('cc_layer', 'name', __('Layers', 'castconductor'));
        $layer_singular = $get_label('cc_layer', 'singular', __('Layer', 'castconductor'));
        register_post_type('cc_layer', array(
            'labels' => array(
                'name' => $layer_name,
                'singular_name' => $layer_singular,
                'add_new' => sprintf(__('Add New %s', 'castconductor'), $layer_singular),
                'add_new_item' => sprintf(__('Add New %s', 'castconductor'), $layer_singular),
                'edit_item' => sprintf(__('Edit %s', 'castconductor'), $layer_singular),
                'new_item' => sprintf(__('New %s', 'castconductor'), $layer_singular),
                'view_item' => sprintf(__('View %s', 'castconductor'), $layer_singular),
                'search_items' => sprintf(__('Search %s', 'castconductor'), $layer_name),
                'not_found' => sprintf(__('No %s found', 'castconductor'), strtolower($layer_name)),
                'not_found_in_trash' => sprintf(__('No %s found in trash', 'castconductor'), strtolower($layer_name)),
            ),
            'public' => false,
            'show_ui' => true,
            'show_in_menu' => 'castconductor',
            'show_in_rest' => true,
            'supports' => array('title', 'editor', 'thumbnail', 'custom-fields'),
            'has_archive' => false,
            'rewrite' => false,
        ));
    }
    
    /**
     * Register taxonomies
     */
    public function register_taxonomies() {
        // Add taxonomies if needed
    }
    
    /**
     * Plugin activation
     */
    public function activate() {
        // Check requirements before activation
        if (!$this->check_requirements()) {
            wp_die(__('CastConductor V5 requires WordPress ' . CASTCONDUCTOR_MIN_WP_VERSION . ' and PHP ' . CASTCONDUCTOR_MIN_PHP_VERSION . ' or higher.', 'castconductor'));
        }
        
        // Load dependencies first
        $this->load_dependencies();
        
        // Initialize database component
        $this->database = new CastConductor_Database();
        
        // Create database tables
        $this->database->create_tables();
        
        // Create analytics tables
        $analytics_db = CC_Analytics_Database::instance();
        $analytics_db->create_tables();
        
        // Schedule analytics cron jobs
        $analytics_agg = CC_Analytics_Aggregation::instance();
        $analytics_agg->schedule_cron_jobs();
        
        // Set default options
        $this->set_default_options();
        
        // Schedule activation wizard ONLY on fresh installs (not updates)
        // Check if setup was already completed - if so, don't show wizard again
        if (!get_option('castconductor_setup_completed')) {
            update_option('castconductor_activation_wizard_pending', true);
        }

        // Bootstrap minimal defaults immediately so the admin canvas is functional on fresh installs.
        // Use output buffering to suppress any inadvertent output from REST-style responses
        ob_start();
        try {
            // Ensure default containers/backgrounds/blocks exist
            if (!$this->database) { $this->database = new CastConductor_Database(); }
            $this->database->create_tables();
            $this->database->create_default_containers();
            $this->database->create_default_backgrounds();
            $blocks = get_option('castconductor_default_content_blocks');
            if (!is_array($blocks) || empty($blocks)) {
                $this->database->create_default_content_blocks();
            } else {
                // Ensure no duplicates exist from prior activations
                if (method_exists($this->database, 'maybe_deduplicate_content_blocks')) {
                    $this->database->maybe_deduplicate_content_blocks();
                }
            }

            // Seed lower-third defaults for OOTB preview
            $lower_id = (int) get_option('castconductor_default_lower_third_id');
            if ($lower_id > 0) {
                $req = new WP_REST_Request('POST', '/castconductor/v5/containers/' . $lower_id . '/defaults');
                $req->set_param('interval', 15);
                $req->set_param('mode', 'full_reset');
                $ctl = new CastConductor_Containers_Controller();
                $ctl->apply_production_defaults($req);
            }

            // Seed Upper Third left/right zones (Location & Time / Weather) once
            if (!$this->activation_wizard) { $this->activation_wizard = new CastConductor_Activation_Wizard(); }
            if (method_exists($this->activation_wizard, 'seed_upper_third_zone_assignments')) {
                $this->activation_wizard->seed_upper_third_zone_assignments();
            }
        } catch (Exception $e) {
            if (defined('WP_DEBUG') && WP_DEBUG) {
                error_log('CastConductor V5 activation bootstrap warning: ' . $e->getMessage());
            }
        }
        ob_end_clean(); // Discard any output generated during bootstrap
        
        // Flush rewrite rules
        flush_rewrite_rules();
    }
    
    /**
     * Set default options on activation
     */
    private function set_default_options() {
        // URL Configuration (from Toaster or manual setup)
        add_option('castconductor_wordpress_domain', home_url());
        add_option('castconductor_stream_url', ''); // No default - must be configured
        add_option('castconductor_metadata_url', ''); // No default - must be configured
        
        // Real data API configurations
        add_option('castconductor_openweather_api_key', ''); // Weather data
        add_option('castconductor_artwork_search_enabled', true); // Album artwork discovery
        add_option('castconductor_geolocation_enabled', true); // IP-based location
        
        // Plugin settings
        add_option('castconductor_version', CASTCONDUCTOR_VERSION);
        add_option('castconductor_first_activation', current_time('mysql'));
        add_option('castconductor_setup_completed', false);
        
        // Data protection settings (default: preserve user data on uninstall)
        add_option('castconductor_preserve_data_on_uninstall', true);
        add_option('castconductor_data_protection_confirmed', false);
        
        // Roku App Configuration
        add_option('castconductor_channel_name', '');
        add_option('castconductor_app_name', '');
        add_option('castconductor_roku_icon_fhd', 0);  // 540×405 PNG
        add_option('castconductor_roku_icon_hd', 0);   // 290×218 PNG
        add_option('castconductor_roku_splash_fhd', 0); // 1920×1080 JPG
        add_option('castconductor_roku_splash_hd', 0);  // 1280×720 JPG
        add_option('castconductor_roku_splash_sd', 0);  // 720×480 JPG
        
        // Default container configuration (1280×720 authoring space)
        add_option('castconductor_default_container_config', array(
            'lower_third' => array(
                'name' => 'Lower Third',
                'type' => 'lower_third_container',
                'position' => array('x' => 0, 'y' => 480, 'width' => 1280, 'height' => 240),
                'enabled' => true
            )
        ));
    }
    
    /**
     * Plugin deactivation
     */
    public function deactivate() {
        // Clean up scheduled events
        wp_clear_scheduled_hook('castconductor_cleanup_expired_content');
        
        // Clear license check schedule
        CastConductor_License_Manager::clear_scheduled_events();
        
        // Clear analytics cron jobs
        CC_Analytics_Aggregation::clear_cron_jobs();
        
        // Flush rewrite rules
        flush_rewrite_rules();
    }
    
    /**
     * Plugin uninstall (static method)
     * 
     * IMPORTANT: Data protection - only removes data if explicitly confirmed by user
     */
    public static function uninstall() {
        // Check if user has explicitly chosen to remove data
        $preserve_data = get_option('castconductor_preserve_data_on_uninstall', true);
        $data_removal_confirmed = get_option('castconductor_data_removal_confirmed', false);
        
        // Default behavior: PRESERVE USER DATA
        if ($preserve_data && !$data_removal_confirmed) {
            // Log the preservation (for development debugging)
            if (defined('WP_DEBUG') && WP_DEBUG) {
                error_log('CastConductor V5: Plugin uninstalled - USER DATA PRESERVED by default');
            }
            
            // Only remove the plugin settings, keep all user content
            delete_option('castconductor_activation_wizard_pending');
            delete_option('castconductor_preserve_data_on_uninstall');
            delete_option('castconductor_data_removal_confirmed');
            delete_option('castconductor_data_protection_confirmed');
            
            return; // Exit early - preserve all user data
        }
        
        // Only reach here if user explicitly confirmed data removal
        if (defined('WP_DEBUG') && WP_DEBUG) {
            error_log('CastConductor V5: Plugin uninstalled - USER CONFIRMED DATA REMOVAL');
        }
        
        global $wpdb;
        
        // Remove database tables (only if explicitly confirmed)
        $tables = array(
            $wpdb->prefix . 'castconductor_containers',
            $wpdb->prefix . 'castconductor_content_blocks',
            $wpdb->prefix . 'castconductor_container_blocks'
        );
        
        foreach ($tables as $table) {
            $wpdb->query("DROP TABLE IF EXISTS {$table}");
        }
        
        // Remove custom post types content (only if explicitly confirmed)
        $post_types = array('cc_shoutout', 'cc_sponsor', 'cc_promo');
        foreach ($post_types as $post_type) {
            $posts = get_posts(array(
                'post_type' => $post_type,
                'numberposts' => -1,
                'post_status' => 'any'
            ));
            
            foreach ($posts as $post) {
                wp_delete_post($post->ID, true); // Force delete
            }
        }
        
        // Remove options
        delete_option('castconductor_version');
        delete_option('castconductor_setup_completed');
        delete_option('castconductor_first_activation');
        delete_option('castconductor_setup_date');
        
        // Remove all castconductor_* options
        $wpdb->query("DELETE FROM {$wpdb->options} WHERE option_name LIKE 'castconductor_%'");
        
        // Remove transients
        $wpdb->query("DELETE FROM {$wpdb->options} WHERE option_name LIKE '_transient_castconductor_%' OR option_name LIKE '_transient_timeout_castconductor_%'");
    }
    
    /**
     * PHP version notice
     */
    public function php_version_notice() {
        echo '<div class="notice notice-error"><p>';
        echo sprintf(__('CastConductor V5 requires PHP %s or higher. You are running PHP %s.', 'castconductor'), CASTCONDUCTOR_MIN_PHP_VERSION, PHP_VERSION);
        echo '</p></div>';
    }
    
    /**
     * WordPress version notice
     */
    public function wp_version_notice() {
        global $wp_version;
        echo '<div class="notice notice-error"><p>';
        echo sprintf(__('CastConductor V5 requires WordPress %s or higher. You are running WordPress %s.', 'castconductor'), CASTCONDUCTOR_MIN_WP_VERSION, $wp_version);
        echo '</p></div>';
    }
}

/**
 * Get the main plugin instance
 */
function castconductor() {
    return CastConductor_Plugin::instance();
}

// Initialize the plugin
castconductor();
