<?php
/**
 * Cast Conductor Proprietary License v5
 * SPDX-License-Identifier: LicenseRef-CastConductor-Proprietary-v5
 * 
 * Copyright (c) 2025 CastConductor.com. All Rights Reserved.
 * 
 * This file is part of Cast Conductor ("Software"). The Software and its source
 * code constitute proprietary, confidential, and trade secret information of
 * CastConductor.com ("Company"). Any access or use is governed strictly by the
 * Cast Conductor Proprietary License v5 ("License"). By installing, copying,
 * accessing, compiling, or otherwise using the Software you agree to be bound by
 * all terms of the License. If you do not agree, you must cease use immediately.
 * 
 * Key Terms (Summary – see full License for binding terms):
 *  1. No Redistribution: You may not publish, distribute, sublicense, rent,
 *     lease, transfer, sell, or otherwise make the Software (or any derivative)
 *     available to any third party without prior written consent of Company.
 *  2. No Modification: Modification, reverse engineering, decompilation, or
 *     disassembly is prohibited except to the limited extent expressly permitted
 *     by applicable law that cannot be contractually waived.
 *  3. Confidentiality: Treat all source code and related artifacts as Company
 *     Confidential Information. Maintain at least the same degree of care as for
 *     your own confidential materials, and not less than reasonable care.
 *  4. No Patent License: No express or implied patent rights are granted. Future
 *     patents (if any) are fully reserved.
 *  5. No Trademark License: Company names, marks, and logos may not be used
 *     without prior written permission.
 *  6. Limited Internal Use: Use is limited solely to internal evaluation and
 *     operation of licensed Cast Conductor deployments. Commercial hosting or
 *     resale as a service requires a separate written agreement.
 *  7. Telemetry & License Validation: The Software may periodically transmit a
 *     hashed installation identifier, domain (or site ID), plugin/app version,
 *     and a truncated (non-reversible) fragment of the license key solely to
 *     validate activation status and enforce licensing. This minimal "phone home"
 *     check contains no personal or content data. If optional telemetry is later
 *     introduced it will be limited to aggregate operational metrics (no PII),
 *     fully documented, and optionally disableable per published instructions.
 *  8. Third-Party Components: The Software may include open source components
 *     covered by their own licenses. See THIRD-PARTY-NOTICES.md. Those licenses
 *     govern their respective components; this License governs all remaining code.
 *  9. Export Compliance: You are responsible for compliance with all applicable
 *     export control and sanctions laws.
 * 10. Warranty Disclaimer: THE SOFTWARE IS PROVIDED "AS IS" WITHOUT WARRANTY OF
 *     ANY KIND, EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO MERCHANTABILITY,
 *     FITNESS FOR A PARTICULAR PURPOSE, TITLE, AND NON-INFRINGEMENT.
 * 11. Limitation of Liability: IN NO EVENT WILL COMPANY OR AUTHORS BE LIABLE FOR
 *     ANY INDIRECT, SPECIAL, INCIDENTAL, CONSEQUENTIAL, EXEMPLARY, OR PUNITIVE
 *     DAMAGES, OR LOST PROFITS, EVEN IF ADVISED OF THE POSSIBILITY.
 * 12. Acceptance: Use of the Software constitutes acceptance of the License.
 * 13. Enforcement: Unauthorized reproduction or distribution may result in civil
 *     and criminal penalties and will be prosecuted to the maximum extent allowed
 *     by law.
 * 
 * Authoritative EULA: EULA-v5.1.md (repository root – private) and https://castconductor.com/eula
 * Precedence: If this summary conflicts with the EULA, the EULA governs.
 * Revision: Current EULA revision v5.1 (subject to update; check EULA for current enterprise thresholds).
 * 
 * Full License text available from: licensing@castconductor.com
 * Security reports: security@castconductor.com
 * Commercial inquiries: licensing@castconductor.com
 * 
 * END OF HEADER
 */

/**
 * Shoutout Moderation Controller
 *
 * Handles the optional shoutout moderation system for admin control
 * over user-submitted content with configurable approval workflows.
 *
 * @package CastConductor
 * @subpackage API
 * @since 5.0.0
 */

if (!defined('ABSPATH')) {
    exit;
}

/**
 * Shoutout Moderation Controller Class
 *
 * Optional moderation system for user-submitted shoutouts with
 * configurable auto-approve vs manual review workflows.
 */
class CastConductor_Shoutout_Moderation_Controller extends WP_REST_Controller {

    /**
     * Register the routes for shoutout moderation
     */
    public function register_routes() {
    $version = '5';
        $namespace = 'castconductor/v' . $version;
        $base = 'shoutout-moderation';

        // Get pending shoutouts for moderation
        register_rest_route($namespace, '/' . $base . '/pending', array(
            array(
                'methods'             => WP_REST_Server::READABLE,
                'callback'            => array($this, 'get_pending_shoutouts'),
                'permission_callback' => array($this, 'check_moderation_permissions'),
                'args'               => array(
                    'page' => array(
                        'validate_callback' => function($param, $request, $key) {
                            return is_numeric($param);
                        }
                    ),
                    'per_page' => array(
                        'validate_callback' => function($param, $request, $key) {
                            return is_numeric($param) && $param <= 100;
                        }
                    ),
                ),
            ),
        ));

        // Approve shoutout
        register_rest_route($namespace, '/' . $base . '/approve/(?P<id>\d+)', array(
            array(
                'methods'             => WP_REST_Server::EDITABLE,
                'callback'            => array($this, 'approve_shoutout'),
                'permission_callback' => array($this, 'check_moderation_permissions'),
                'args'               => array(
                    'id' => array(
                        'validate_callback' => function($param, $request, $key) {
                            return is_numeric($param);
                        }
                    ),
                ),
            ),
        ));

        // Reject shoutout
        register_rest_route($namespace, '/' . $base . '/reject/(?P<id>\d+)', array(
            array(
                'methods'             => WP_REST_Server::EDITABLE,
                'callback'            => array($this, 'reject_shoutout'),
                'permission_callback' => array($this, 'check_moderation_permissions'),
                'args'               => array(
                    'id' => array(
                        'validate_callback' => function($param, $request, $key) {
                            return is_numeric($param);
                        }
                    ),
                    'reason' => array(
                        'required' => false,
                        'type' => 'string',
                    ),
                ),
            ),
        ));

        // Bulk moderation actions
        register_rest_route($namespace, '/' . $base . '/bulk', array(
            array(
                'methods'             => WP_REST_Server::EDITABLE,
                'callback'            => array($this, 'bulk_moderation'),
                'permission_callback' => array($this, 'check_moderation_permissions'),
                'args'               => array(
                    'action' => array(
                        'required' => true,
                        'enum' => array('approve', 'reject', 'delete'),
                    ),
                    'ids' => array(
                        'required' => true,
                        'type' => 'array',
                    ),
                    'reason' => array(
                        'required' => false,
                        'type' => 'string',
                    ),
                ),
            ),
        ));

        // Get moderation statistics
        register_rest_route($namespace, '/' . $base . '/stats', array(
            array(
                'methods'             => WP_REST_Server::READABLE,
                'callback'            => array($this, 'get_moderation_stats'),
                'permission_callback' => array($this, 'check_moderation_permissions'),
            ),
        ));

        // Update moderation settings
        register_rest_route($namespace, '/' . $base . '/settings', array(
            array(
                'methods'             => WP_REST_Server::READABLE,
                'callback'            => array($this, 'get_moderation_settings'),
                'permission_callback' => array($this, 'check_moderation_permissions'),
            ),
            array(
                'methods'             => WP_REST_Server::EDITABLE,
                'callback'            => array($this, 'update_moderation_settings'),
                'permission_callback' => array($this, 'check_moderation_permissions'),
                'args'               => $this->get_moderation_settings_args(),
            ),
        ));

        // Test reCAPTCHA configuration
        register_rest_route($namespace, '/' . $base . '/test-recaptcha', array(
            array(
                'methods'             => WP_REST_Server::CREATABLE,
                'callback'            => array($this, 'test_recaptcha'),
                'permission_callback' => array($this, 'check_moderation_permissions'),
                'args'               => array(
                    'test_token' => array(
                        'required' => true,
                        'type' => 'string',
                    ),
                ),
            ),
        ));
    }

    /**
     * Get pending shoutouts for moderation
     *
     * @param WP_REST_Request $request Full data about the request.
     * @return WP_Error|WP_REST_Response
     */
    public function get_pending_shoutouts($request) {
        global $wpdb;

        // Check if moderation is enabled
        if (!get_option('castconductor_shoutout_moderation_enabled', true)) {
            return new WP_Error(
                'moderation_disabled',
                'Shoutout moderation is currently disabled.',
                array('status' => 400)
            );
        }

        try {
            $page = $request->get_param('page') ?: 1;
            $per_page = $request->get_param('per_page') ?: 20;
            $offset = ($page - 1) * $per_page;

            $table_name = $wpdb->prefix . 'castconductor_content_blocks';
            
            // Get pending shoutouts
            $shoutouts = $wpdb->get_results($wpdb->prepare(
                "SELECT id, name, content, metadata, created_at, updated_at 
                 FROM {$table_name} 
                 WHERE type = 'shoutout' 
                 AND status = 'pending' 
                 ORDER BY created_at DESC 
                 LIMIT %d OFFSET %d",
                $per_page,
                $offset
            ));

            // Get total count for pagination
            $total = $wpdb->get_var(
                "SELECT COUNT(*) FROM {$table_name} 
                 WHERE type = 'shoutout' AND status = 'pending'"
            );

            // Parse metadata for each shoutout
            foreach ($shoutouts as &$shoutout) {
                $shoutout->metadata = json_decode($shoutout->metadata, true);
                $shoutout->content = json_decode($shoutout->content, true);
                
                // Add moderation context
                $shoutout->moderation_context = $this->get_moderation_context($shoutout);
            }

            return new WP_REST_Response(array(
                'success' => true,
                'shoutouts' => $shoutouts,
                'pagination' => array(
                    'page' => $page,
                    'per_page' => $per_page,
                    'total' => (int) $total,
                    'total_pages' => ceil($total / $per_page),
                ),
            ), 200);

        } catch (Exception $e) {
            return new WP_Error(
                'database_error',
                'Failed to retrieve pending shoutouts: ' . $e->getMessage(),
                array('status' => 500)
            );
        }
    }

    /**
     * Approve a shoutout
     *
     * @param WP_REST_Request $request Full data about the request.
     * @return WP_Error|WP_REST_Response
     */
    public function approve_shoutout($request) {
        global $wpdb;

        $shoutout_id = (int) $request['id'];

        try {
            $table_name = $wpdb->prefix . 'castconductor_content_blocks';
            
            // Update shoutout status to approved
            $result = $wpdb->update(
                $table_name,
                array(
                    'status' => 'approved',
                    'updated_at' => current_time('mysql')
                ),
                array('id' => $shoutout_id, 'type' => 'shoutout'),
                array('%s', '%s'),
                array('%d', '%s')
            );

            if ($result === false) {
                return new WP_Error(
                    'approval_failed',
                    'Failed to approve shoutout.',
                    array('status' => 500)
                );
            }

            if ($result === 0) {
                return new WP_Error(
                    'shoutout_not_found',
                    'Shoutout not found or already processed.',
                    array('status' => 404)
                );
            }

            // Log moderation action
            $this->log_moderation_action($shoutout_id, 'approved', get_current_user_id());

            // Send notification if enabled
            $this->send_moderation_notification($shoutout_id, 'approved');

            return new WP_REST_Response(array(
                'success' => true,
                'message' => 'Shoutout approved successfully.',
                'shoutout_id' => $shoutout_id,
                'status' => 'approved'
            ), 200);

        } catch (Exception $e) {
            return new WP_Error(
                'approval_error',
                'Failed to approve shoutout: ' . $e->getMessage(),
                array('status' => 500)
            );
        }
    }

    /**
     * Reject a shoutout
     *
     * @param WP_REST_Request $request Full data about the request.
     * @return WP_Error|WP_REST_Response
     */
    public function reject_shoutout($request) {
        global $wpdb;

        $shoutout_id = (int) $request['id'];
        $reason = sanitize_text_field($request->get_param('reason') ?: 'Inappropriate content');

        try {
            $table_name = $wpdb->prefix . 'castconductor_content_blocks';
            
            // Update shoutout status to rejected
            $result = $wpdb->update(
                $table_name,
                array(
                    'status' => 'rejected',
                    'updated_at' => current_time('mysql')
                ),
                array('id' => $shoutout_id, 'type' => 'shoutout'),
                array('%s', '%s'),
                array('%d', '%s')
            );

            if ($result === false) {
                return new WP_Error(
                    'rejection_failed',
                    'Failed to reject shoutout.',
                    array('status' => 500)
                );
            }

            if ($result === 0) {
                return new WP_Error(
                    'shoutout_not_found',
                    'Shoutout not found or already processed.',
                    array('status' => 404)
                );
            }

            // Log moderation action with reason
            $this->log_moderation_action($shoutout_id, 'rejected', get_current_user_id(), $reason);

            return new WP_REST_Response(array(
                'success' => true,
                'message' => 'Shoutout rejected successfully.',
                'shoutout_id' => $shoutout_id,
                'status' => 'rejected',
                'reason' => $reason
            ), 200);

        } catch (Exception $e) {
            return new WP_Error(
                'rejection_error',
                'Failed to reject shoutout: ' . $e->getMessage(),
                array('status' => 500)
            );
        }
    }

    /**
     * Handle bulk moderation actions
     *
     * @param WP_REST_Request $request Full data about the request.
     * @return WP_Error|WP_REST_Response
     */
    public function bulk_moderation($request) {
        global $wpdb;

        $action = $request->get_param('action');
        $ids = $request->get_param('ids');
        $reason = sanitize_text_field($request->get_param('reason') ?: 'Bulk action');

        if (empty($ids) || !is_array($ids)) {
            return new WP_Error(
                'invalid_ids',
                'No valid shoutout IDs provided.',
                array('status' => 400)
            );
        }

        try {
            $table_name = $wpdb->prefix . 'castconductor_content_blocks';
            $processed = 0;
            $errors = array();

            foreach ($ids as $id) {
                $id = (int) $id;
                if ($id <= 0) continue;

                $new_status = ($action === 'approve') ? 'approved' : 
                             (($action === 'reject') ? 'rejected' : 'deleted');

                if ($action === 'delete') {
                    $result = $wpdb->delete(
                        $table_name,
                        array('id' => $id, 'type' => 'shoutout'),
                        array('%d', '%s')
                    );
                } else {
                    $result = $wpdb->update(
                        $table_name,
                        array(
                            'status' => $new_status,
                            'updated_at' => current_time('mysql')
                        ),
                        array('id' => $id, 'type' => 'shoutout'),
                        array('%s', '%s'),
                        array('%d', '%s')
                    );
                }

                if ($result !== false && $result > 0) {
                    $processed++;
                    $this->log_moderation_action($id, $action, get_current_user_id(), $reason);
                } else {
                    $errors[] = "Failed to process shoutout ID: {$id}";
                }
            }

            return new WP_REST_Response(array(
                'success' => true,
                'message' => "Bulk {$action} completed.",
                'processed' => $processed,
                'total' => count($ids),
                'action' => $action,
                'errors' => $errors
            ), 200);

        } catch (Exception $e) {
            return new WP_Error(
                'bulk_action_error',
                'Failed to process bulk action: ' . $e->getMessage(),
                array('status' => 500)
            );
        }
    }

    /**
     * Get moderation statistics
     *
     * @param WP_REST_Request $request Full data about the request.
     * @return WP_Error|WP_REST_Response
     */
    public function get_moderation_stats($request) {
        global $wpdb;

        try {
            $table_name = $wpdb->prefix . 'castconductor_content_blocks';
            
            // Get counts by status
            $stats = $wpdb->get_results(
                "SELECT status, COUNT(*) as count 
                 FROM {$table_name} 
                 WHERE type = 'shoutout' 
                 GROUP BY status"
            );

            $formatted_stats = array(
                'pending' => 0,
                'approved' => 0,
                'rejected' => 0,
                'total' => 0
            );

            foreach ($stats as $stat) {
                $formatted_stats[$stat->status] = (int) $stat->count;
                $formatted_stats['total'] += (int) $stat->count;
            }

            // Get today's stats
            $today_stats = $wpdb->get_results($wpdb->prepare(
                "SELECT status, COUNT(*) as count 
                 FROM {$table_name} 
                 WHERE type = 'shoutout' 
                 AND DATE(created_at) = %s 
                 GROUP BY status",
                current_time('Y-m-d')
            ));

            $today_formatted = array(
                'pending' => 0,
                'approved' => 0,
                'rejected' => 0,
                'total' => 0
            );

            foreach ($today_stats as $stat) {
                $today_formatted[$stat->status] = (int) $stat->count;
                $today_formatted['total'] += (int) $stat->count;
            }

            // Get moderation settings
            $moderation_enabled = get_option('castconductor_shoutout_moderation_enabled', true);
            $auto_approve = get_option('castconductor_shoutout_auto_approve', false);

            return new WP_REST_Response(array(
                'success' => true,
                'stats' => array(
                    'all_time' => $formatted_stats,
                    'today' => $today_formatted,
                ),
                'settings' => array(
                    'moderation_enabled' => $moderation_enabled,
                    'auto_approve' => $auto_approve,
                ),
                'generated_at' => current_time('mysql')
            ), 200);

        } catch (Exception $e) {
            return new WP_Error(
                'stats_error',
                'Failed to retrieve moderation statistics: ' . $e->getMessage(),
                array('status' => 500)
            );
        }
    }

    /**
     * Get moderation settings
     *
     * @param WP_REST_Request $request Full data about the request.
     * @return WP_Error|WP_REST_Response
     */
    public function get_moderation_settings($request) {
        try {
            $settings = array(
                'moderation_enabled' => get_option('castconductor_shoutout_moderation_enabled', true),
                'auto_approve' => get_option('castconductor_shoutout_auto_approve', false),
                'word_filter' => get_option('castconductor_shoutout_word_filter', true),
                'duplicate_check' => get_option('castconductor_shoutout_duplicate_check', true),
                'recaptcha_enabled' => get_option('castconductor_recaptcha_enabled', true),
                'recaptcha_site_key' => get_option('castconductor_recaptcha_site_key', ''),
                'recaptcha_secret_key' => get_option('castconductor_recaptcha_secret_key', ''),
                'recaptcha_threshold' => get_option('castconductor_recaptcha_threshold', 0.5),
                'email_notifications' => get_option('castconductor_moderation_email_notifications', true),
                'notification_email' => get_option('castconductor_moderation_notification_email', get_option('admin_email')),
            );

            return new WP_REST_Response(array(
                'success' => true,
                'settings' => $settings
            ), 200);

        } catch (Exception $e) {
            return new WP_Error(
                'settings_error',
                'Failed to retrieve moderation settings: ' . $e->getMessage(),
                array('status' => 500)
            );
        }
    }

    /**
     * Update moderation settings
     *
     * @param WP_REST_Request $request Full data about the request.
     * @return WP_Error|WP_REST_Response
     */
    public function update_moderation_settings($request) {
        try {
            $settings = $request->get_json_params();

            // Update each setting with validation
            if (isset($settings['moderation_enabled'])) {
                update_option('castconductor_shoutout_moderation_enabled', (bool) $settings['moderation_enabled']);
            }

            if (isset($settings['auto_approve'])) {
                update_option('castconductor_shoutout_auto_approve', (bool) $settings['auto_approve']);
            }

            if (isset($settings['word_filter'])) {
                update_option('castconductor_shoutout_word_filter', (bool) $settings['word_filter']);
            }

            if (isset($settings['duplicate_check'])) {
                update_option('castconductor_shoutout_duplicate_check', (bool) $settings['duplicate_check']);
            }

            if (isset($settings['recaptcha_enabled'])) {
                update_option('castconductor_recaptcha_enabled', (bool) $settings['recaptcha_enabled']);
            }

            if (isset($settings['recaptcha_site_key'])) {
                update_option('castconductor_recaptcha_site_key', sanitize_text_field($settings['recaptcha_site_key']));
            }

            if (isset($settings['recaptcha_secret_key'])) {
                update_option('castconductor_recaptcha_secret_key', sanitize_text_field($settings['recaptcha_secret_key']));
            }

            if (isset($settings['recaptcha_threshold'])) {
                $threshold = floatval($settings['recaptcha_threshold']);
                $threshold = max(0.0, min(1.0, $threshold)); // Clamp between 0 and 1
                update_option('castconductor_recaptcha_threshold', $threshold);
            }

            if (isset($settings['email_notifications'])) {
                update_option('castconductor_moderation_email_notifications', (bool) $settings['email_notifications']);
            }

            if (isset($settings['notification_email'])) {
                $email = sanitize_email($settings['notification_email']);
                if (is_email($email)) {
                    update_option('castconductor_moderation_notification_email', $email);
                }
            }

            return new WP_REST_Response(array(
                'success' => true,
                'message' => 'Moderation settings updated successfully.',
                'updated_settings' => $settings
            ), 200);

        } catch (Exception $e) {
            return new WP_Error(
                'settings_update_error',
                'Failed to update moderation settings: ' . $e->getMessage(),
                array('status' => 500)
            );
        }
    }

    /**
     * Test reCAPTCHA configuration
     *
     * @param WP_REST_Request $request Full data about the request.
     * @return WP_Error|WP_REST_Response
     */
    public function test_recaptcha($request) {
        $test_token = $request->get_param('test_token');
        $secret_key = get_option('castconductor_recaptcha_secret_key');

        if (empty($secret_key)) {
            return new WP_Error(
                'no_secret_key',
                'reCAPTCHA secret key not configured.',
                array('status' => 400)
            );
        }

        try {
            // Verify with Google reCAPTCHA API
            $response = wp_remote_post('https://www.google.com/recaptcha/api/siteverify', array(
                'body' => array(
                    'secret' => $secret_key,
                    'response' => $test_token,
                    'remoteip' => $_SERVER['REMOTE_ADDR'] ?? ''
                )
            ));

            if (is_wp_error($response)) {
                return new WP_Error(
                    'recaptcha_request_failed',
                    'Failed to contact reCAPTCHA service: ' . $response->get_error_message(),
                    array('status' => 500)
                );
            }

            $body = wp_remote_retrieve_body($response);
            $result = json_decode($body, true);

            return new WP_REST_Response(array(
                'success' => true,
                'recaptcha_result' => $result,
                'valid' => $result['success'] ?? false,
                'score' => $result['score'] ?? null,
                'action' => $result['action'] ?? null,
                'error_codes' => $result['error-codes'] ?? array()
            ), 200);

        } catch (Exception $e) {
            return new WP_Error(
                'recaptcha_test_error',
                'Failed to test reCAPTCHA: ' . $e->getMessage(),
                array('status' => 500)
            );
        }
    }

    /**
     * Get moderation context for a shoutout
     *
     * @param object $shoutout Shoutout data
     * @return array Moderation context
     */
    private function get_moderation_context($shoutout) {
        $context = array(
            'flags' => array(),
            'risk_level' => 'low',
            'recommendations' => array()
        );

        // Check for potential issues
        $content = $shoutout->content['message'] ?? '';
        $name = $shoutout->content['name'] ?? '';

        // Word filter check
        if (get_option('castconductor_shoutout_word_filter', true)) {
            $flagged_words = $this->check_word_filter($content . ' ' . $name);
            if (!empty($flagged_words)) {
                $context['flags'][] = 'word_filter';
                $context['risk_level'] = 'high';
                $context['flagged_words'] = $flagged_words;
            }
        }

        // Duplicate check
        if (get_option('castconductor_shoutout_duplicate_check', true)) {
            if ($this->check_duplicate_content($content)) {
                $context['flags'][] = 'duplicate_content';
                $context['risk_level'] = 'medium';
            }
        }

        // Add recommendations
        if (empty($context['flags'])) {
            $context['recommendations'][] = 'Content appears clean - safe to approve';
        } else {
            $context['recommendations'][] = 'Review flagged content before approval';
        }

        return $context;
    }

    /**
     * Check content against word filter
     *
     * @param string $content Content to check
     * @return array Flagged words
     */
    private function check_word_filter($content) {
        // Basic word filter - in production, this would be more sophisticated
        $blocked_words = array(
            'spam', 'fake', 'scam', 'hack', 'virus', 'phishing',
            // Add more as needed
        );

        $flagged = array();
        $content_lower = strtolower($content);

        foreach ($blocked_words as $word) {
            if (strpos($content_lower, $word) !== false) {
                $flagged[] = $word;
            }
        }

        return $flagged;
    }

    /**
     * Check for duplicate content
     *
     * @param string $content Content to check
     * @return bool True if duplicate found
     */
    private function check_duplicate_content($content) {
        global $wpdb;

        $table_name = $wpdb->prefix . 'castconductor_content_blocks';
        
        $count = $wpdb->get_var($wpdb->prepare(
            "SELECT COUNT(*) FROM {$table_name} 
             WHERE type = 'shoutout' 
             AND JSON_EXTRACT(content, '$.message') = %s 
             AND status IN ('approved', 'pending')
             AND created_at > DATE_SUB(NOW(), INTERVAL 24 HOUR)",
            $content
        ));

        return (int) $count > 0;
    }

    /**
     * Log moderation action
     *
     * @param int $shoutout_id Shoutout ID
     * @param string $action Action taken
     * @param int $user_id User ID who performed action
     * @param string $reason Optional reason
     */
    private function log_moderation_action($shoutout_id, $action, $user_id, $reason = '') {
        // Log moderation actions for audit trail
        // In production, this might go to a separate log table
        error_log(sprintf(
            'CastConductor Moderation: User %d performed %s on shoutout %d. Reason: %s',
            $user_id,
            $action,
            $shoutout_id,
            $reason
        ));
    }

    /**
     * Send moderation notification
     *
     * @param int $shoutout_id Shoutout ID
     * @param string $action Action taken
     */
    private function send_moderation_notification($shoutout_id, $action) {
        if (!get_option('castconductor_moderation_email_notifications', true)) {
            return;
        }

        $email = get_option('castconductor_moderation_notification_email', get_option('admin_email'));
        
        // Send notification email (simplified implementation)
        wp_mail(
            $email,
            "CastConductor: Shoutout {$action}",
            "Shoutout ID {$shoutout_id} has been {$action}."
        );
    }

    /**
     * Get moderation settings arguments
     *
     * @return array Arguments
     */
    private function get_moderation_settings_args() {
        return array(
            'moderation_enabled' => array(
                'required' => false,
                'type' => 'boolean',
            ),
            'auto_approve' => array(
                'required' => false,
                'type' => 'boolean',
            ),
            'word_filter' => array(
                'required' => false,
                'type' => 'boolean',
            ),
            'duplicate_check' => array(
                'required' => false,
                'type' => 'boolean',
            ),
            'recaptcha_enabled' => array(
                'required' => false,
                'type' => 'boolean',
            ),
            'recaptcha_site_key' => array(
                'required' => false,
                'type' => 'string',
            ),
            'recaptcha_secret_key' => array(
                'required' => false,
                'type' => 'string',
            ),
            'recaptcha_threshold' => array(
                'required' => false,
                'type' => 'number',
                'minimum' => 0.0,
                'maximum' => 1.0,
            ),
            'email_notifications' => array(
                'required' => false,
                'type' => 'boolean',
            ),
            'notification_email' => array(
                'required' => false,
                'type' => 'string',
                'format' => 'email',
            ),
        );
    }

    /**
     * Check permissions for moderation operations
     * Allows: Administrators and Editors (edit_posts)
     *
     * @param WP_REST_Request $request Full data about the request.
     * @return WP_Error|bool
     */
    public function check_moderation_permissions($request) {
        return current_user_can('manage_options') || current_user_can('edit_posts');
    }
}
